// BookList.cs
//
// Copyright (C) 2004  Peter Knowles
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330,
// Boston, MA 02111-1307, USA.

using System;
using System.Collections;
using System.Text;
using System.IO;

namespace OpenLibrie.BookList {

  /// <summary>
  /// Represents a collection of books
  /// </summary>
  class BookList {

    /// <summary>
    /// Array of BookListItems contained in the index
    /// </summary>
    private ArrayList bookListItems;

	// <summary>
    /// Array of BookListItems contained in the index
    /// </summary>
    public ArrayList Books {
      get { return bookListItems; }
    }
    
    /// <summary>
    /// Initializes a new empty BookList
    /// </summary>
    public BookList() {
      this.bookListItems = new ArrayList();
    }
    
    /// <summary>
    /// Initializes a new BookList from file
    /// </summary>
    /// <param name="filename">
    /// BookList filename
    /// </param>
    public BookList(String filename) {
    	LoadFromFile(filename);
    }
    
    /// <summary>
    /// Loads a new BookList from a file
    /// </summary>
    /// <param name="filename">
    /// BookList file to load
    /// </param>
    public void LoadFromFile(String filename) {
    	this.bookListItems = new ArrayList();
      	long fileSize = new FileInfo(filename).Length;

      	/* Used to store the whole booklist file as bytes */
      	byte[] bookListBytes = new byte[fileSize];

      	/* Read the entire LRF file into bookListBytes */
      	byte[] tempBuffer = new Byte[4096];
      	FileStream fs = new FileStream(filename, FileMode.Open);
      	int numBytesRead = 0;
      	int curPosition = 0;
      	while (0 != (numBytesRead = fs.Read(tempBuffer, 0, 4096))) {
			Buffer.BlockCopy(tempBuffer, 0, bookListBytes, curPosition, numBytesRead);
			curPosition += numBytesRead;
      	}
      	fs.Close ();
      	tempBuffer = null;
      	
      	/* Go through the booklist data, item by item, and generate the BookListItem objects */
      	curPosition = 0;
      	int bookListItemSize;
      	while(curPosition < fileSize) {
      		bookListItemSize = BitConverter.ToInt16(bookListBytes, curPosition);
      		byte[] bookListItemBytes = new byte[bookListItemSize];
      		Buffer.BlockCopy(bookListBytes, curPosition, bookListItemBytes, 0, bookListItemSize);
      		bookListItems.Add(new BookListItem(bookListItemBytes));
      		curPosition += bookListItemSize;
      	}
    }
    
    /// <summary>
    /// Add a BookListItem into the list
    /// </summary>
    /// <param name="item">
    /// BookList item to add
    /// </param>
    public void AddItem(BookListItem item) {
      bookListItems.Add(item);
    }

    /// <summary>
    /// Save the booklist into a file.
    /// </summary>
    /// <param name="filename">
    /// Filename of booklist file to save to
    /// </param>
    public void SaveToFile(string filename, Boolean sortByTitle, Boolean verbose) {
      byte[] indexData = GenerateIndex(sortByTitle, verbose);

      FileStream fs = File.Create(filename);
      fs.Write(indexData, 0, indexData.Length);
      fs.Close();
    }
    
    /// <summary>
    /// Generate a byte[] which represents the booklist of all BookListItems
    /// </summary>
    public byte[] GenerateIndex(Boolean sortByTitle, Boolean verbose) {
      byte[] index;
      int totalLength = 0;
      ArrayList byteArrays = new ArrayList();

      /* Sort the BookItems alphabetically by title if requested */
      if (sortByTitle == true) {
	IComparer bookSort = new BookListAlphabeticalTitleSort();
	this.bookListItems.Sort(bookSort);
      }

      /* First generate all the byte[]'s for the items and stick them into byteArrays */
      foreach (BookListItem bookListItem in bookListItems) {
	if (verbose == true) {
	  Console.WriteLine("Adding: {0} ({1})", bookListItem.ID, bookListItem.Title);
	}
	byte[] tempBytes = bookListItem.GenerateIndexEntry();
	totalLength += tempBytes.Length;
	byteArrays.Add(tempBytes);
      }

      /* Allocate the required memory space for the index */
      index = new Byte[totalLength];

      /* Loop through all the byte[]'s in byteArrays and copy them into the index */
      int curPosition = 0;
      foreach (byte[] itemBytes in byteArrays) {
	Buffer.BlockCopy(itemBytes, 0, index, curPosition, itemBytes.Length);
	curPosition += itemBytes.Length;
      }

      return index;
    }
  }
  
}
