// BookListItem.cs
//
// Copyright (C) 2004  Peter Knowles
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330,
// Boston, MA 02111-1307, USA.

using System;
using System.Text;
using System.Xml;
using System.IO;
using System.Collections;

using OpenLibrie.LRFLib;

namespace OpenLibrie.BookList {

  /// <summary>
  /// Represents an item in a BookList (i.e. A single book)
  /// </summary>
  class BookListItem {

    /// <summary>
    /// Size of a byte
    /// </summary>
    public const int BYTE_SIZE = 1;

    /// <summary>
    /// Size of a WORD
    /// </summary>
    public const int WORD_SIZE = 2;

    /// <summary>
    /// Size of a DWORD
    /// </summary>
    public const int DWORD_SIZE = 4;

    /// <summary>
    /// Offset of total length in index file
    /// </summary>
    public const int INDEX_OFFSET_TOTAL_LEN = 0x00;

    /// <summary>
    /// Offset of length of title in index file
    /// </summary>
    public const int INDEX_OFFSET_TITLE_LEN = 0x02;

    /// <summary>
    /// Offset of length of title reading in index file
    /// </summary>
    public const int INDEX_OFFSET_TITLEREADING_LEN = 0x04;

    /// <summary>
    /// Offset of length of author in index file
    /// </summary>
    public const int INDEX_OFFSET_AUTHOR_LEN = 0x06;

    /// <summary>
    /// Offset of length of author reading in index file
    /// </summary>
    public const int INDEX_OFFSET_AUTHORREADING_LEN = 0x08;

    /// <summary>
    /// Offset of length of publisher in index file
    /// </summary>
    public const int INDEX_OFFSET_PUBLISHER_LEN = 0x0A;

    /// <summary>
    /// Offset of length of file location in index file
    /// </summary>
    public const int INDEX_OFFSET_LOCATION_LEN = 0x0C;

    /// <summary>
    /// Offset of length of GIF data in index file
    /// </summary>
    public const int INDEX_OFFSET_GIF_LEN = 0x0E;

    /// <summary>
    /// Offset of length of second magic sequence in index file
    /// </summary>
    public const int INDEX_OFFSET_MAGICSEQUENCE2_LEN = 0x10;

	/// <summary>
    /// Start of the data (follows after the header)
    /// </summary>
    public const int INDEX_OFFSET_DATA_START = 0x21;

    private string id, date, title, titleReading, author, authorReading;
    private string location, publisher;
    private long fileSize;
    private byte thumbnailType;
    private Byte[] magicSequence1, magicSequence2, gifData;

    /// <summary>
    /// The ID of the book
    /// </summary>
    public string ID {
      get { return id; }
      set { id = value; }
    }

    /// <summary>
    /// The creation date of the LRF file
    /// </summary>
    public string Date {
      get { return date; }
      set { date = value; }
    }

    /// <summary>
    /// The title of the LRF file
    /// </summary>
    public string Title {
      get { return title; }
      set { title = value; }
    }

    /// <summary>
    /// The reading of the title
    /// </summary>
    public string TitleReading {
      get { return titleReading; }
      set { titleReading = value; }
    }

    /// <summary>
    /// The author of the LRF file
    /// </summary>
    public string Author {
      get { return author; }
      set { author = value; }
    }

    /// <summary>
    /// The reading of the author
    /// </summary>
    public string AuthorReading {
      get { return authorReading; }
      set { authorReading = value; }
    }

    /// <summary>
    /// The location of the LRF file (relative to the memory stick root)
    /// </summary>
    public string Location {
      get { return location; }
      set { location = value; }
    }

    /// <summary>
    /// The GIF data for the LRF thumbnail image
    /// </summary>
    public byte[] GIFData {
      get { return gifData; }
      set { gifData = value; }
    }

    /// <summary>
    /// The publisher of the LRF file
    /// </summary>
    public string Publisher {
      get { return publisher; }
      set { publisher = value; }
    }

    /// <summary>
    /// The size of the LRF file
    /// </summary>
    public long FileSize {
      get { return fileSize; }
      set { fileSize = value; }
    }

	/// <summary>
	/// The image type of the thumbnail
	/// </summary>
	public byte ThumbnailType {
		get { return thumbnailType; }
		set { thumbnailType = value; }
	}

	/// <summary>
	/// Initializes a new instance of the BookListItem class, using data from
	/// the byte array.
	/// </summary>
	/// <param name="bookListItemBytes">
	/// The byte array representing the BookListItem
	/// </param>
	public BookListItem(byte[] bookListBytes) {
		UnicodeEncoding unicode = new UnicodeEncoding();
		ASCIIEncoding ascii = new ASCIIEncoding();

		SetupMagic();
	
		/* Set default values */
    	this.id = "";
      	this.author = "";
      	this.authorReading = "";
      	this.title = "";
      	this.titleReading = "";
      	this.date = "";
      	this.publisher = "";
      
      	/* Get Field Lengths */
		int bookListItemSize = BitConverter.ToInt16(bookListBytes, INDEX_OFFSET_TOTAL_LEN);
    	int titleLength = BitConverter.ToInt16(bookListBytes, INDEX_OFFSET_TITLE_LEN);
    	int titleReadingLength = BitConverter.ToInt16(bookListBytes, INDEX_OFFSET_TITLEREADING_LEN);
    	int authorLength = BitConverter.ToInt16(bookListBytes, INDEX_OFFSET_AUTHOR_LEN);
    	int authorReadingLength = BitConverter.ToInt16(bookListBytes, INDEX_OFFSET_AUTHORREADING_LEN);
    	int publisherLength = BitConverter.ToInt16(bookListBytes, INDEX_OFFSET_PUBLISHER_LEN);
    	int locationLength = BitConverter.ToInt16(bookListBytes, INDEX_OFFSET_LOCATION_LEN);
    	int gifLength = BitConverter.ToInt16(bookListBytes, INDEX_OFFSET_GIF_LEN);
        int magicSequence2Length = BitConverter.ToInt16(bookListBytes, INDEX_OFFSET_MAGICSEQUENCE2_LEN);
        
        int curPosition = INDEX_OFFSET_DATA_START;
        
        /* Get the ID */
        this.ID = unicode.GetString(bookListBytes, curPosition, 32);
        curPosition += 32;
        
        /* Get the ID */
        this.Date = ascii.GetString(bookListBytes, curPosition, 15);
        curPosition += 15;
        
        /* Get the Title */
        this.Title = unicode.GetString(bookListBytes, curPosition, titleLength);
        curPosition += titleLength;
        
        /* Get the Title Reading */
        this.TitleReading = unicode.GetString(bookListBytes, curPosition, titleReadingLength);
        curPosition += titleReadingLength;
        
        /* Get the Author */
        this.Author = unicode.GetString(bookListBytes, curPosition, authorLength);
        curPosition += authorLength;
        
        /* Get the Author Reading */
        this.AuthorReading = unicode.GetString(bookListBytes, curPosition, authorReadingLength);
        curPosition += authorReadingLength;
        
        /* Get the Publisher */
        this.Publisher = unicode.GetString(bookListBytes, curPosition, publisherLength);
        curPosition += publisherLength;
        
        /* Get the Location */
        this.Location = ascii.GetString(bookListBytes, curPosition, locationLength);
        curPosition += locationLength;
        
        /* Get the LRF File Size */
        this.FileSize = BitConverter.ToInt32(bookListBytes, curPosition);
        curPosition += 4;
        
        /* Skip the 16 NULL bytes */
        curPosition += 16;
        
        /* Get the Thumbnail Type */
        this.ThumbnailType = bookListBytes[curPosition];
        curPosition += 1;
        
        /* Get the GIF data */
      	this.gifData = new byte[gifLength];
      	Buffer.BlockCopy(bookListBytes, curPosition, this.gifData, 0, gifLength);
	}

    /// <summary>
    /// Initializes a new instance of the BookListItem class, using data from
    /// the specified file.
    /// </summary>
    /// <param name="lrffile">
    /// The LRFFile class from which to read XWL Info and GIF Image
    /// </param>
    /// <param name="fileLocation">
    /// Location of LRF file relative to the memory stick root.
    /// </param>
    /// <param name="dump">
    /// If true, dump the thumbnail image and XML info block.
    /// </param>
    public BookListItem(LRFFile lrfFile, string fileLocation, Boolean dump) {
      /* Convert the path to uppercase, but leave the filename alone */
      int indexLastSlash = fileLocation.LastIndexOf(Path.DirectorySeparatorChar);
      string onlyPath = fileLocation.Substring(0,indexLastSlash + 1);
      string onlyFile = fileLocation.Substring(indexLastSlash + 1);
      this.location = "a:" + onlyPath.ToUpper() + onlyFile;

      /* Convert to UNIX style separator, that's what we need */
      this.location = this.location.Replace("\\","/");

      SetupMagic();

      /* Set default values */
      this.id = "";
      this.author = "";
      this.authorReading = "";
      this.title = "";
      this.titleReading = "";
      this.date = "";
      this.publisher = "";

      /* Get the LRF size */
      this.fileSize = lrfFile.FileSize;

      /* Copy GIF Thumnail image from LRFFile */
      this.gifData = new byte[lrfFile.GIFData.Length];
      Buffer.BlockCopy(lrfFile.GIFData, 0, this.gifData, 0, lrfFile.GIFData.Length);

      #region ParseXMLInfoBlock
      /* Parse the XML Info block from LRFFile to pull out the title,author,date and ID */
      try {
	XmlTextReader textReader = new XmlTextReader(new MemoryStream(lrfFile.XMLInfoBlock));
	textReader.MoveToContent();
	while (textReader.Read()) {
	  /* Only bother with this node if it's an element containing data */
	  if ((textReader.NodeType == XmlNodeType.Element) && (! textReader.IsEmptyElement)) {
	    switch (textReader.Name) {
	      case "BookID":
		this.id = textReader.ReadElementString();
		break;
	      case "Author":
		if (textReader.GetAttribute("reading") != null) {
		  this.authorReading = textReader.GetAttribute("reading");
		}
	        this.author = textReader.ReadElementString();
		break;
	      case "Title":
		if (textReader.GetAttribute("reading") != null) {
		  this.titleReading = textReader.GetAttribute("reading");
		}
	        this.title = textReader.ReadElementString();
	        break;
	      case "CreationDate":
		this.date = textReader.ReadElementString().Trim();
	        break;
	      case "Publisher":
		this.publisher = textReader.ReadElementString().Trim();
		break;
	    }
	  }
	} 
      } catch (Exception e) {
	throw new XMLInfoParseException(e.Message);
      }
      #endregion

      #region DumpXMLAndThumbNail
      if (dump == true) {
	/* Dump the thumbnail */
	string thumbDumpFilename = onlyFile + ".booklistgen_thumb.gif";
	FileStream thumbFile = File.Create(thumbDumpFilename);
	thumbFile.Write(this.gifData, 0, this.gifData.Length);
	thumbFile.Close();

	/* Dump the XML Info */
	string xmlDumpFilename = onlyFile + ".booklistgen_info.xml";
	FileStream xmlFile = File.Create(xmlDumpFilename);
	xmlFile.Write(lrfFile.XMLInfoBlock, 0, lrfFile.XMLInfoBlock.Length);
	xmlFile.Close();
      }
      #endregion
    }

    /// <summary>
    /// This sets some magic values up. They still need to be decyphered at some point.
    /// </summary>
    private void SetupMagic() {
      /* This seems to be the last part of the header */
      magicSequence1 = new Byte[] {0x60,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
				   0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00 };

      /* This comes between the file location and the GIF data */
      magicSequence2 = new Byte[] {0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
				   0x00,0x00,0x00,0x00,0x00,0x00,0x00};
    }

   
    /// <summary>
    /// Generates a representation of the item that can be used in the BookIndex
    /// </summary>
    /// <returns>
    /// byte[] which can be used in the BookIndex
    /// </returns>
    public Byte[] GenerateIndexEntry() {
      UnicodeEncoding unicode = new UnicodeEncoding();
      ASCIIEncoding ascii = new ASCIIEncoding();

      /* GIF Data and length */
      Byte[] bytesGIFData = gifData;
      Byte[] bytesGIFLength = BitConverter.GetBytes(gifData.Length);

      /* Book ID (needs to be padded/truncated to 32 bytes) and length */
      Byte[] bytesIDTemp = unicode.GetBytes(id);
      Byte[] bytesID = new Byte[32];
      if (bytesIDTemp.Length > 32) {
	Buffer.BlockCopy(bytesIDTemp, 0, bytesID, 0, 32);
      } else {
	Buffer.BlockCopy(bytesIDTemp, 0, bytesID, 0, bytesIDTemp.Length);
      }
      Byte[] bytesIDLength = BitConverter.GetBytes(bytesID.Length);


      /* Date (needs to be padded/truncated to 15 characters) and length  */
      Byte[] bytesDateTemp = ascii.GetBytes(date);
      Byte[] bytesDate = new Byte[15];
      if (bytesDateTemp.Length > 15) {
	Buffer.BlockCopy(bytesDateTemp, 0, bytesDate, 0, 15);
      } else {
	Buffer.BlockCopy(bytesDateTemp, 0, bytesDate, 0, bytesDateTemp.Length);
      }
      Byte[] bytesDateLength = BitConverter.GetBytes(bytesDate.Length);

      /* Title and length */
      Byte[] bytesTitle = unicode.GetBytes(title);
      Byte[] bytesTitleLength = BitConverter.GetBytes(bytesTitle.Length);

      /* Title reading and length */
      Byte[] bytesTitleReading = unicode.GetBytes(titleReading);
      Byte[] bytesTitleReadingLength = BitConverter.GetBytes(bytesTitleReading.Length);

      /* Author and length */
      Byte[] bytesAuthor = unicode.GetBytes(author);
      Byte[] bytesAuthorLength = BitConverter.GetBytes(bytesAuthor.Length);

      /* Author reading and length */
      Byte[] bytesAuthorReading = unicode.GetBytes(authorReading);
      Byte[] bytesAuthorReadingLength = BitConverter.GetBytes(bytesAuthorReading.Length);

      /* Publisher and length */
      Byte[] bytesPublisher = unicode.GetBytes(publisher);
      Byte[] bytesPublisherLength = BitConverter.GetBytes(bytesPublisher.Length);

      /* Location and length */
      Byte[] bytesLocation = ascii.GetBytes(location);
      Byte[] bytesLocationLength = BitConverter.GetBytes(bytesLocation.Length);

      /* File size */
      Byte[] bytesFileSize = BitConverter.GetBytes((int)this.fileSize);

      /* Magic sequence 1 and length */
      Byte[] bytesMagicSequence1 = magicSequence1;
      Byte[] bytesMagicSequence1Length = BitConverter.GetBytes(bytesMagicSequence1.Length);

      /* Magic sequence 2 and length */
      Byte[] bytesMagicSequence2 = magicSequence2;
      Byte[] bytesMagicSequence2Length = BitConverter.GetBytes(bytesMagicSequence2.Length);

      /* Length of the header, including the magic sequence */
      int headerLength = (WORD_SIZE * 8) + magicSequence1.Length;

      /* Total index length, including header and all data */
      long totalLength = headerLength + bytesID.Length + bytesDate.Length + 
	                 bytesTitle.Length + bytesTitleReading.Length + bytesAuthor.Length + 
                         bytesAuthorReading.Length + bytesPublisher.Length + 
                         bytesLocation.Length +  bytesFileSize.Length + 
                         bytesMagicSequence2.Length + BYTE_SIZE + gifData.Length;
      Byte[] byteStream = new Byte[totalLength];
      Byte[] bytesTotalLength = BitConverter.GetBytes(totalLength);

      /* Copy all the header data into place. The holes will be set to null */
      Buffer.BlockCopy(bytesTotalLength, 0, byteStream, INDEX_OFFSET_TOTAL_LEN, WORD_SIZE);
      Buffer.BlockCopy(bytesTitleLength, 0, byteStream, INDEX_OFFSET_TITLE_LEN, WORD_SIZE);
      Buffer.BlockCopy(bytesTitleReadingLength, 0, byteStream, INDEX_OFFSET_TITLEREADING_LEN, WORD_SIZE);
      Buffer.BlockCopy(bytesAuthorLength, 0, byteStream, INDEX_OFFSET_AUTHOR_LEN, WORD_SIZE);
      Buffer.BlockCopy(bytesAuthorReadingLength, 0, byteStream, INDEX_OFFSET_AUTHORREADING_LEN, WORD_SIZE);
      Buffer.BlockCopy(bytesPublisherLength, 0, byteStream, INDEX_OFFSET_PUBLISHER_LEN, WORD_SIZE);
      Buffer.BlockCopy(bytesLocationLength, 0, byteStream, INDEX_OFFSET_LOCATION_LEN, WORD_SIZE);
      Buffer.BlockCopy(bytesGIFLength, 0, byteStream, INDEX_OFFSET_GIF_LEN, WORD_SIZE);

      /* Stick the first magic sequence onto the end of the header */
      Buffer.BlockCopy(bytesMagicSequence1, 0, byteStream, INDEX_OFFSET_MAGICSEQUENCE2_LEN, bytesMagicSequence1.Length);

      /* Everything from here on is writen linearly, with the current position advancing
         as we go */
      int curPosition = headerLength;

      /* Write the ID */
      Buffer.BlockCopy(bytesID, 0, byteStream, curPosition, bytesID.Length);
      curPosition += bytesID.Length;

      /* Write the date */
      Buffer.BlockCopy(bytesDate, 0, byteStream, curPosition, bytesDate.Length);
      curPosition += bytesDate.Length;

      /* Write the title */ 
      Buffer.BlockCopy(bytesTitle, 0, byteStream, curPosition, bytesTitle.Length);
      curPosition += bytesTitle.Length;
     
      /* Write the title reading */ 
      Buffer.BlockCopy(bytesTitleReading, 0, byteStream, curPosition, bytesTitleReading.Length);
      curPosition += bytesTitleReading.Length;

      /* Write the author */
      Buffer.BlockCopy(bytesAuthor, 0, byteStream, curPosition, bytesAuthor.Length);
      curPosition += bytesAuthor.Length;

      /* Write the author reading */
      Buffer.BlockCopy(bytesAuthorReading, 0, byteStream, curPosition, bytesAuthorReading.Length);
      curPosition += bytesAuthorReading.Length;

      /* Write the publisher */
      Buffer.BlockCopy(bytesPublisher, 0, byteStream, curPosition, bytesPublisher.Length);
      curPosition += bytesPublisher.Length;

      /* Write the file location */
      Buffer.BlockCopy(bytesLocation, 0, byteStream, curPosition, bytesLocation.Length);
      curPosition += bytesLocation.Length;

      /* Write the file size */
      Buffer.BlockCopy(bytesFileSize, 0, byteStream, curPosition, bytesFileSize.Length);
      curPosition += bytesFileSize.Length;

      /* Write the second magic sequence */
      Buffer.BlockCopy(bytesMagicSequence2, 0, byteStream, curPosition, bytesMagicSequence2.Length);
      curPosition += bytesMagicSequence2.Length;
      
      /* Only do this if we actually have a thumbnail image */
      if (bytesGIFData.Length > 3) {
	/* Write the Image Type: 0x12 for PNG, 0x14 for GIF, 0x13 for BM and 0x10 for JPEG */
	if ((bytesGIFData[1] == 'P') && (bytesGIFData[2] == 'N') && (bytesGIFData[3] == 'G')) {
	  /* It's a PNG */
	  byteStream[curPosition] = 0x12;
	} else if ((bytesGIFData[0] == 'B') && (bytesGIFData[1] == 'M')) {
	  /* It's a BM */
	  byteStream[curPosition] = 0x13;
	} else if ((bytesGIFData[6] == 'J') && (bytesGIFData[7] == 'F') && 
		   (bytesGIFData[8] == 'I') && (bytesGIFData[9] == 'F')) {
	  /* It's a JFIF */
	  byteStream[curPosition] = 0x11;
	} else if ((bytesGIFData[0] == 'G') && (bytesGIFData[1] == 'I') && 
		   (bytesGIFData[2] == 'F') && (bytesGIFData[3] == '8')) {
	  /* It's a GIF */
	  byteStream[curPosition] = 0x14;
	} else {
	  /* I give up, lets just call it a GIF and hope nobody notices */
	  byteStream[curPosition] = 0x14;
	}
      }
      curPosition += BYTE_SIZE;

      /* Write the GIF data */
      Buffer.BlockCopy(bytesGIFData, 0, byteStream, curPosition, bytesGIFData.Length);

      return byteStream;
    }
  }

  /* Class used to sort the books by alphabetical order by title */
  class BookListAlphabeticalTitleSort : IComparer {
    public int Compare (Object bookA, Object bookB) {
      return ((BookListItem)bookA).Title.CompareTo(((BookListItem)bookB).Title);
    }
  }
  
  /* Class used to sort the books by alphabetical order by author */
  class BookListAlphabeticalAuthorSort : IComparer {
    public int Compare (Object bookA, Object bookB) {
      return ((BookListItem)bookA).Author.CompareTo(((BookListItem)bookB).Author);
    }
  }  
}
