// BookListGenGUI.cs
//
// Copyright (C) 2005  Peter Knowles
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330,
// Boston, MA 02111-1307, USA.

using System;
using Gtk;
using Gdk;
using Glade;
using OpenLibrie.BookList;
using OpenLibrie.LRFLib;
using OpenLibrie.Utils;
using System.IO;
using System.Collections;
using System.Xml;
using System.Reflection;
using System.Text;
using System.Security;

namespace OpenLibrie {

  /// <summary>
  /// A very simplistic class used to represent the config for the GUI
  /// </summary>
  public class BookListGenGUIConfig {
  
    /// <summary>
    /// The filename of the configuration file
    /// </summary>
    private string configFile;
    
    /// <summary>
    /// Returns a string from the configuration
    /// </summary>
    /// <param name="key">
    /// The XPATH of the XML node to retrieve
    /// </param>
    /// <returns>
    /// The requested value from the config
    /// </returns>
    public string GetString (string key) {
      return Convert.ToString(GetValue(key, typeof(string)));
    }
    
    /// <summary>
    /// Returns an integer from the configuration
    /// </summary>
    /// <param name="key">
    /// The XPATH of the XML node to retrieve
    /// </param>
    /// <returns>
    /// The requested value from the config
    /// </returns>
    public uint GetInt (string key) {
      return Convert.ToUInt16(GetValue(key, typeof(int)));
    }
    
    /// <summary>
    /// Generates a config object from the specified config file
    /// </summary>
    /// <param name="confFilename">
    /// The configuration filename
    /// </param>
    public BookListGenGUIConfig(string confFilename) {
      this.configFile = confFilename;
    }
    
    /// <summary>
    /// Returns an object from the configuration file
    /// </summary>
    /// <param name="key">
    /// The XPATH of the XML node to retrieve
    /// </param>
    /// <param name="resultType">
    /// The type of object to return
    /// </param>
    /// <returns>
    /// The requested value from the config
    /// </returns>
    private object GetValue (string key, System.Type resultType) {
      XmlNode selectedNode;
      XmlDocument document = new XmlDocument();
      object result = "";
      document.Load( this.configFile );
      string requestedNode = key.Substring(0, key.LastIndexOf("//"));
      
      if ( (selectedNode = document.SelectSingleNode(requestedNode) ) != null ) {
	XmlElement target = (XmlElement)selectedNode.SelectSingleNode(key.Replace(requestedNode,""));
	if ( target != null ) {
	  result = target.GetAttribute("value");
	}
      }
      /* Return the result in the desired type */
      if (resultType == typeof(int)) {
	return Convert.ToUInt32(result);
      } else {
	return Convert.ToString(result);
      }
    }
    
    /// <summary>
    /// Create the configuration file (and directory if necessary)
    /// </summary>
    public void Create () {
      /* Create the directory and any parents necessary */
      if (! Directory.Exists(Path.GetDirectoryName(this.configFile))) {
	Directory.CreateDirectory(Path.GetDirectoryName(this.configFile));
      }
      /* Write out a minimal XML file with BookListGen root element */
      XmlTextWriter writer = new XmlTextWriter( this.configFile , null );
      writer.Formatting = Formatting.Indented;
      writer.WriteStartDocument();
      writer.WriteStartElement("BookListGen");
      writer.WriteAttributeString("version", "1.0");
      writer.WriteStartElement("Preferences");
      writer.WriteEndElement();
      writer.Flush();
      writer.Close();
    }
    
    /// <summary>
    /// Sets a key in the configuration file
    /// </summary>
    /// <param name="key">
    /// The XPATH of the XML node to set
    /// </param>
    /// <param name="val">
    /// The value to set
    /// </param>
    public void SetValue (string key, string val) {
      XmlNode selectedNode;
      XmlDocument document = new XmlDocument();
      document.Load( this.configFile );
      
      /* Find the correct node */
      string requestedNode = key.Substring(0, key.LastIndexOf("//"));
      selectedNode =  document.SelectSingleNode(requestedNode);
      if( selectedNode == null ) {
	throw new Exception("Could not select XML node in config file");
      }
      XmlElement target = (XmlElement)selectedNode.SelectSingleNode(key.Replace(requestedNode,""));
      
      if (target != null) {
	/* The key already exists, so just set the attribute of the existing node */
	target.SetAttribute("value", val);
      } else {
	/* Generate a new element node and set the attribute */
	requestedNode = key.Substring(key.LastIndexOf("//")+2);
	XmlElement newEntry = document.CreateElement(requestedNode.Substring(0, requestedNode.IndexOf("[@")).Trim());
	requestedNode = requestedNode.Substring(requestedNode.IndexOf("'")+1);
	newEntry.SetAttribute("key", requestedNode.Substring(0, requestedNode.IndexOf("'")) );
	newEntry.SetAttribute("value", val);
	selectedNode.AppendChild(newEntry);
      }
      /* Write out the XML configuration to disk */
      XmlTextWriter writer = new XmlTextWriter( this.configFile , null );
      writer.Formatting = Formatting.Indented;
      document.WriteTo( writer );
      writer.Flush();
      writer.Close();
    }
    
  } /* Class: BookListGenGUIConfig */
  
  /// <summary>
  /// Class representing the BookListGen GUI
  /// </summary>
  public class BookListGenGUI {
    ///private Glade.XML mainXML;
    
    /// <summary>
    /// The current LibrieFileStructure being used.
    /// </summary>
    private LibrieFileStructure librieFileStructure;
    
    /// <summary>
    /// Save the list to disk as changes are made?
    /// (This needs to be turned off when we refresh everything
    ///  to avoid saving too many times)
    /// </summary>
    private bool realTimeUpdate;
    
    /// <summary>
    /// Sort order: 0 for None, 1 for Title, 2 for Author
    /// </summary>
    private uint sortType;
    
    /// <summary>
    /// The current root directory of the memory stick
    /// </summary>
    private string librieRootDir;
    
    /// <summary>
    /// The current configuration object
    /// </summary>
    private BookListGenGUIConfig config;
    
    /// <summary>
    /// Have we saved a config? (If not, we need to force the
    /// preferences window on the user)
    /// </summary>
    private bool configSaved;
    
    /// <summary>
    /// Are we editing the XML of a corrupt LRF file?
    /// </summary>
    private bool editingCorruptXML;

    /// <summary>
    /// Filename of corrupt XML file on the memory stick.
    /// </summary>
    private string corruptDestXMLFile;

    /// <summary>
    /// Filename of original corrupt LRF file.
    /// </summary>
    private string corruptXMLFile;

    /// <summary>
    /// The currently selected book
    /// </summary>
    private TreeIter selectedBookIterator;

    /// <summary>
    /// The preferences window
    /// </summary>
    [Glade.Widget]
      Gtk.Window preferencesWindow;
    
    /// <summary>
    /// The main window (Containing the BookList)
    /// </summary>
    [Glade.Widget]
      Gtk.Window bookListWindow;
    
    /// <summary>
    /// The close button on the main window
    /// </summary>
    [Glade.Widget]
      Gtk.Button closeButton;
    
    /// <summary>
    /// The add button on the main window
    /// </summary>
    [Glade.Widget]
      Gtk.Button addButton;
    
    /// <summary>
    /// The remove button on the main window.
    /// </summary>
    [Glade.Widget]
      Gtk.Button removeButton;
    
    /// <summary>
    /// The refresh button on the main window.
    /// </summary>
    [Glade.Widget]
      Gtk.Button refreshButton;
    
    /// <summary>
    /// The preferences button on the main window.
    /// </summary>
    [Glade.Widget]
      Gtk.Button preferencesButton;
    
    /// <summary>
    /// The booklist location label on the main window
    /// </summary>
    [Glade.Widget]
      Gtk.Label bookListLocationLabel;
    
    /// <summary>
    /// The BookList on the main window
    /// </summary>
    [Glade.Widget]
      Gtk.TreeView booklist;
    
    /// <summary>
    /// The librie root directory entry on the preferences window
    /// </summary>
    [Glade.Widget]
      Gtk.Entry librieRootEntry;
    
    /// <summary>
    /// The browse button on the preferences window
    /// </summary>
    [Glade.Widget]
      Gtk.Button browseRootDirButton;
    
    /// <summary>
    /// The create filesystem button on the preferences window
    /// </summary>
    [Glade.Widget]
      Gtk.Button createFilesystemButton;
    
    /// <summary>
    /// The cancel button on the preferences window
    /// </summary>
    [Glade.Widget]
      Gtk.Button preferencesCancelButton;
    
    /// <summary>
    /// The OK button on the preferences window
    /// </summary>
    [Glade.Widget]
      Gtk.Button preferencesOkButton;
    
    /// <summary>
    /// The sort order option on the preferences window
    /// </summary>
    [Glade.Widget]
      Gtk.OptionMenu bookListSortOrder;

    /// <summary>
    /// The Edit LRF Meta Information window
    /// </summary>
    [Glade.Widget]
    Gtk.Window editLRFWindow;

    /// <summary>
    /// The Edit thumbnail entry box
    /// </summary>
    [Glade.Widget]
    Gtk.Entry thumbnailEditEntry;

    /// <summary>
    /// The Edit author entry box
    /// </summary>
    [Glade.Widget]
    Gtk.Entry authorEditEntry;

    /// <summary>
    /// The Edit author reading entry box
    /// </summary>
    [Glade.Widget]
    Gtk.Entry authorReadingEditEntry;

    /// <summary>
    /// The Edit title entry box
    /// </summary>
    [Glade.Widget]
    Gtk.Entry titleEditEntry;

    /// <summary>
    /// The Edit title reading entry box
    /// </summary>
    [Glade.Widget]
    Gtk.Entry titleReadingEditEntry;

    /// <summary>
    /// The Edit description entry box
    /// </summary>
    [Glade.Widget]
    Gtk.Entry descriptionEditEntry;

    /// <summary>
    /// The Edit creation date entry box
    /// </summary>
    [Glade.Widget]
    Gtk.Entry creationDateEditEntry;

    /// <summary>
    /// The Edit publisher entry box
    /// </summary>
    [Glade.Widget]
    Gtk.Entry publisherEditEntry;

    /// <summary>
    /// The Edit BookID entry box
    /// </summary>
    [Glade.Widget]
    Gtk.Entry bookIDEditEntry;

    /// <summary>
    /// The Edit browse for thumbnail button
    /// </summary>
    [Glade.Widget]
    Gtk.Button browseEditButton;

    /// <summary>
    /// The Edit cancel button
    /// </summary>
    [Glade.Widget]
    Gtk.Button cancelEditButton;

    /// <summary>
    /// The Edit ok button
    /// </summary>
    [Glade.Widget]
    Gtk.Button okEditButton;

    /// <summary>
    /// The Edit description label
    /// </summary>
    [Glade.Widget]
    Gtk.Label descriptionEditLabel;

    /// <summary>
    /// The Edit XML Window
    /// </summary>
    [Glade.Widget]
    Gtk.Window xmlEditWindow;

    /// <summary>
    /// The cancel XML Edit Button
    /// </summary>
    [Glade.Widget]
    Gtk.Button cancelXMLEditButton;

    /// <summary>
    /// The OK XML Edit Button
    /// </summary>
    [Glade.Widget]
    Gtk.Button okXMLEditButton;

    /// <summary>
    /// The XML Text View
    /// </summary>
    [Glade.Widget]
    Gtk.TextView xmlEditTextView;

    /// <summary>
    /// The Edit XML Information button
    /// </summary>
    [Glade.Widget]
    Gtk.Button editFullXMLButton;

    /// <summary>
    /// Run the GUI
    /// </summary>
    /// <param name="args">
    /// Command line arguments
    /// </param>
    public static void Main (string[] args) {
      new BookListGenGUI(args);
    }
    
    /// <summary>
    /// Construct and run the GUI
    /// </summary>
    /// <param name="args">
    /// Command line arguments
    /// </param>
    public BookListGenGUI (string[] args)  {
      Application.Init();
      
      /* Set the default configuration filename (eg. ~/.config/BookListGen on Linux) */
      string configFile = Environment.GetFolderPath (Environment.SpecialFolder.ApplicationData) + Path.DirectorySeparatorChar + 
	((Assembly.GetEntryAssembly()).GetName()).Name + Path.DirectorySeparatorChar + ((Assembly.GetEntryAssembly()).GetName()).Name + ".xml";
      config = new BookListGenGUIConfig(configFile);
      
      Glade.XML gxml = new Glade.XML (null, "booklistgengui.glade", null, null);
      gxml.Autoconnect (this);
      
      /* Do some initial configuration of the widgets */
      setupBookList();
      
      /* Have we already got a config file? */
      if (! File.Exists(configFile) ) {
	/* No config, so we need to force the preferences window */
	configSaved = false;
	
	/* Tell the user that this is the initial run */
	MessageDialog md = new MessageDialog (bookListWindow, DialogFlags.DestroyWithParent, MessageType.Info, ButtonsType.Close, 
					      "It looks like this is the first time that you've run BookListGen. Please configure the root directory for the memory stick.");
	md.Run ();
	md.Destroy();
	
				/* Show the preferences, and hide the BookList window */
	preferencesWindow.Show();
	bookListWindow.Hide();
      } else {
	/* We have a config, so read it in. */
	configSaved = true;
	librieRootDir = config.GetString("//BookListGen//Preferences//set[@key='rootdir']");
	sortType = config.GetInt("//BookListGen//Preferences//set[@key='sortorder']");
	librieFileStructure = new LibrieFileStructure(librieRootDir);
	
	/* Refresh the BookList display */
	initializeList();
	
	/* Show the BookList window, and hide the preferences */
	preferencesWindow.Hide();
	bookListWindow.ShowAll();
      }
      /* Enter the main event loop */
      Application.Run();
    }
    
    /// <summary>
    /// Save the BookList to disk
    /// </summary>
    private void updateBookList() {
      TreeIter iter;
      ListStore listStore = (ListStore)(booklist.Model);
      OpenLibrie.BookList.BookList bookList = new OpenLibrie.BookList.BookList();
      bool notDone = listStore.GetIterFirst(out iter);
      
      /* Cycle through the books in our current list, and assemble a new BookList object */
      while (notDone == true) {
	BookListItem item = (BookListItem)listStore.GetValue(iter, 3);
	bookList.AddItem(item);
	notDone = listStore.IterNext(ref iter);
      }
      
      /* Save the BookList, and pop up an error if it goes wrong */
      try {
	bookList.SaveToFile(librieFileStructure.IndexFilename, false, false);
      } catch (Exception e) {
	MessageDialog md = new MessageDialog (bookListWindow, DialogFlags.DestroyWithParent, MessageType.Error, ButtonsType.Close, 
					      "Error occurred while trying to write BookList: " + e.Message );
	md.Run ();
	md.Destroy();
	return;
      }
    }
    
    /// <summary>
    /// Read the BookList off disk, and refresh the BookList display
    /// </summary>
    private void initializeList() {
      bookListLocationLabel.Markup = DisplayString("<b>BookList on " + librieFileStructure.BasePath + "</b>");
      ListStore store = (ListStore)(booklist.Model);
      
      /* We need to kill the updates during the clear. If we don't, then
	 we're going to get a whole load of update calls, and end up saving
	 the list to disk every time */
      realTimeUpdate = false;
      store.Clear();
      realTimeUpdate = true;
      
      /* Read the BookList into memory, and show error if it fails */
      OpenLibrie.BookList.BookList bookList;
      try {
	bookList = new OpenLibrie.BookList.BookList(librieFileStructure.IndexFilename);
      } catch (Exception e) {
	MessageDialog md = new MessageDialog (bookListWindow, DialogFlags.DestroyWithParent, MessageType.Error, ButtonsType.Close, 
					      "Error occurred while trying to read BookList: " + e.Message );
	md.Run ();
	md.Destroy();
	return;
      }
      
      /* Disable the realtime update again */
      realTimeUpdate = false;
      
      /* Sort the BookList according to the specified preference */
      switch (sortType) {
        case 1:
	  /* Sort by Title */
	  IComparer titleSort = new BookListAlphabeticalTitleSort();
	  bookList.Books.Sort(titleSort);
	  break;
        case 2:
	  /* Sort by Author */
	  IComparer authorSort = new BookListAlphabeticalAuthorSort();
	  bookList.Books.Sort(authorSort);
	  break;
        default:
	  /* No sorting */
	  break;
      }
      
      foreach (BookListItem testBook in bookList.Books) {
	/* Get the LRF thumbnail, if there is one */
	Gdk.Pixbuf pb = null;
	if (testBook.GIFData.Length > 0) {
	  pb = new Gdk.Pixbuf(new MemoryStream(testBook.GIFData));
	}
	
	/* Convert the LRF filesize into a readable MB/KB string */
	long fileSize = testBook.FileSize;
	string fileSizeDisplay;
	if (fileSize < (1024 * 1024)) {
	  fileSize = (fileSize / 1024);
	  fileSizeDisplay = fileSize.ToString() + " KB";
	} else {
	  fileSizeDisplay = String.Format("{0:F2} MB", ((float)fileSize / 1024.00 / 1024.00));
	}
	
	/* Strip the a: off the start of the location */
	string location = testBook.Location;
	if (location.StartsWith("a:")) {
	  location = location.Substring(2);
	}
	
	/* Add it to the list */
	store.AppendValues(pb, DisplayString("<b>" + SecurityElement.Escape(testBook.Title) + "</b>\n" + SecurityElement.Escape(testBook.Author) + "\n" + SecurityElement.Escape(location)), fileSizeDisplay, testBook);
      }
      
      /* Turn on the realtime updates and save the BookList to disk */
      realTimeUpdate = true;
      updateBookList();
    }
    
    /// <summary>
    /// Do some initial setup of the widgets.
    /// </summary>
    private void setupBookList() {
      /* Nothing is selected, so we can't remove anything */
      removeButton.Sensitive = false;
      
      /* We can remove more than one book at a time */
      booklist.Selection.Mode = SelectionMode.Multiple;
      
      /* Set up the thumbnail column */
      TreeViewColumn thumbnailColumn = new TreeViewColumn();
      CellRenderer thumbnailRenderer = new CellRendererPixbuf();
      thumbnailColumn.Title = "Thumbnail";
      thumbnailColumn.PackStart(thumbnailRenderer, true);
      thumbnailColumn.AddAttribute(thumbnailRenderer, "pixbuf", 0);
      thumbnailColumn.MinWidth = 70;
      thumbnailColumn.SortOrder = Gtk.SortType.Ascending;
      
      /* Set up the book information column */
      TreeViewColumn bookInfoColumn = new TreeViewColumn();
      CellRenderer bookInfoRenderer = new CellRendererText();
      bookInfoColumn.Title = "Book Info";
      bookInfoColumn.PackStart(bookInfoRenderer, true);
      bookInfoColumn.AddAttribute(bookInfoRenderer, "markup", 1);
      bookInfoColumn.Sizing = TreeViewColumnSizing.Autosize;
      bookInfoColumn.SortOrder = Gtk.SortType.Ascending;
      
      /* Set up the filesize column */
      TreeViewColumn fileSizeColumn = new TreeViewColumn();
      CellRenderer fileSizeRenderer = new CellRendererText();
      /* We add this spacer to the cell to keep a row height of 90, and to push
	 the filesile to the end */
      CellRenderer spaceRenderer = new CellRendererText();
      spaceRenderer.Height = 90;
      fileSizeColumn.Title = "File Size";
      fileSizeColumn.PackStart(spaceRenderer, true);
      fileSizeColumn.PackStart(fileSizeRenderer, false);
      fileSizeColumn.AddAttribute(fileSizeRenderer, "text", 2);
      fileSizeColumn.Alignment = 1F;
          	
      /* Add the columns */
      booklist.AppendColumn (thumbnailColumn);
      booklist.AppendColumn (bookInfoColumn);
      booklist.AppendColumn (fileSizeColumn);
            
      /* Add the list store */
      ListStore store = new ListStore (typeof (Gdk.Pixbuf), typeof (string), typeof (string), typeof (BookListItem));
      booklist.Model = store;
            
            /* Notify us when the list changes, so that we can save */
      booklist.Selection.Changed += new System.EventHandler(OnSelectionChanged);
      booklist.Model.RowsReordered += new Gtk.RowsReorderedHandler(OnBookListOrderChanged);
      booklist.Model.RowDeleted += new Gtk.RowDeletedHandler(OnBookListRowDeleted);
    }
    
    /* Convert from UTF-16 to UTF-8 */
    public string DisplayString(string InputString) {
      if ((Environment.OSVersion.ToString().IndexOf("Unix")) != -1) {
	return InputString;
      } else {
 	byte[] arrItemText = Encoding.UTF8.GetBytes(InputString);
 	return Encoding.Default.GetString(arrItemText);  
      }
    }
      
    /* Convert from UTF-8 to UTF-16 */
    public string AcceptString(string InputString) {
      if ((Environment.OSVersion.ToString().IndexOf("Unix")) != -1) {
	return InputString;
      } else {
	byte[] arrItemText = Encoding.Default.GetBytes(InputString);
	byte[] newBytes = Encoding.Convert(Encoding.UTF8, Encoding.Unicode, arrItemText);
	return Encoding.Unicode.GetString(newBytes);
      }
    }

    /* CALLBACKS FOLLOW */
       
    /* Called when the edit full xml button is clicked */
    public void OnEditFullXMLButtonClicked (object o, EventArgs args) {
      /* Get the currently selected book */
      TreeModel model = booklist.Model;      
      BookListItem bookListItem = (BookListItem)((ListStore)model).GetValue (this.selectedBookIterator, 3);

      int lastSlashIndex = bookListItem.Location.LastIndexOf("/");
      string onlyFilename = bookListItem.Location.Substring(lastSlashIndex + 1);

      if (! File.Exists(librieFileStructure.LRFDirectory + Path.DirectorySeparatorChar + onlyFilename)) {
	MessageDialog mdLRFError = new MessageDialog (editLRFWindow, DialogFlags.DestroyWithParent, MessageType.Error, 
						      ButtonsType.Close, 
						      "Could not locate the LRF file for the selected book. Unable to edit XML information.");
	mdLRFError.Run();
	mdLRFError.Destroy();
	return;
      } else {
	/* Open the LRF file, and read in the XML */
	LRFFile lrfFile = new LRFFile(librieFileStructure.LRFDirectory + Path.DirectorySeparatorChar + onlyFilename);

	/* Fill the TextView in with the XML Information */
	Gtk.TextBuffer xmlTextBuffer = new TextBuffer(null);
      
	/* Populate the text view box with the XML (minus the first two bytes if we're on Windows) */
	if ((Environment.OSVersion.ToString().IndexOf("Unix")) != -1) {
	  xmlTextBuffer.Text = DisplayString(Encoding.Unicode.GetString(lrfFile.XMLInfoBlock));
	} else {
	  xmlTextBuffer.Text = DisplayString(Encoding.Unicode.GetString(lrfFile.XMLInfoBlock).Substring(1));
	}
	
	xmlEditTextView.Buffer = xmlTextBuffer;
	editingCorruptXML = false;
	xmlEditWindow.Show();
      }
    }

    /* Called when the edit full xml window is deleted/closed */
    public void OnXMLEditWindowDeleteEvent (object o, DeleteEventArgs args) {
      xmlEditWindow.Hide();
      args.RetVal = true;
    }

    /* Called when the cancel XML edit button is clicked */
    public void OnCancelXMLEditButtonClicked (object o, EventArgs args) {
      // If we're editing corrupt XML, we better confirm first.
      if (editingCorruptXML) {
	MessageDialog mdSaveXML = new MessageDialog (xmlEditWindow, DialogFlags.DestroyWithParent, MessageType.Warning, ButtonsType.YesNo, 
						   DisplayString("This LRF file has corrupt XML. Are you sure you want to quit without fixing it?") );
	ResponseType resultSaveXML = (ResponseType)mdSaveXML.Run ();
	mdSaveXML.Destroy();
	if (resultSaveXML == ResponseType.No) {
	  return;
	} else {
	  xmlEditWindow.Hide();
	  Application.Quit ();
	}
      } else {
	xmlEditWindow.Hide();
      }
    }

    /* Called when the OK XML edit button is clicked */
    public void OnOKXMLEditButtonClicked (object o, EventArgs args) {
      byte[] xmlUnicodeBytes = Encoding.Unicode.GetBytes("\uFEFF" + AcceptString(xmlEditTextView.Buffer.Text));

      /* Validate the XML to ensure it's well formed before continuing */
      try {
	XmlTextReader verifyXMLReader = new XmlTextReader(new MemoryStream(xmlUnicodeBytes));
	verifyXMLReader.MoveToContent();
	/* Read the entire XML file */
	while (verifyXMLReader.Read()) {
	}
      } catch (Exception xmlException) {
	/* XML is not well formed */
	MessageDialog xmlWellFormedException = new MessageDialog (xmlEditWindow, DialogFlags.DestroyWithParent, MessageType.Error, ButtonsType.Cancel, 
						   DisplayString("XML could not be validated: " + xmlException.Message ) );
	xmlWellFormedException.Run ();
	xmlWellFormedException.Destroy();
	return;
      }

      /* Confirm that the user wants to write the XML to the BookList/LRF File */
      MessageDialog mdSaveXML = new MessageDialog (xmlEditWindow, DialogFlags.DestroyWithParent, MessageType.Warning, ButtonsType.YesNo, 
						   DisplayString("Are you sure you want to update the LRF file with this XML Meta-Information?") );
      ResponseType resultSaveXML = (ResponseType)mdSaveXML.Run ();
      mdSaveXML.Destroy();
      if (resultSaveXML == ResponseType.No) {
	return;
      }

      if (editingCorruptXML) {
	// We're editing a corrupt XML when a new book is added.
	LRFFile destLRFFile = new LRFFile(corruptDestXMLFile);
	LRFFile sourceLRFFile = new LRFFile(corruptXMLFile);

	destLRFFile.XMLInfoBlock = xmlUnicodeBytes;
	sourceLRFFile.XMLInfoBlock = xmlUnicodeBytes;

	byte[] sourceLRFData = sourceLRFFile.getAsBytes();
	byte[] destLRFData = destLRFFile.getAsBytes();

	/* Save the Source LRF file */
	FileStream sourcefs = File.Create(corruptXMLFile);
	sourcefs.Write(sourceLRFData, 0, sourceLRFData.Length);
	sourcefs.Close();

	/* Save the Destination LRF file */
	FileStream destfs = File.Create(corruptDestXMLFile);
	destfs.Write(destLRFData, 0, destLRFData.Length);
	destfs.Close();

	xmlEditWindow.Hide();
	Application.Quit ();
      } else {
	/* Get the selected book */
	TreeModel model = booklist.Model;
	BookListItem bookListItem = (BookListItem)((ListStore)model).GetValue (this.selectedBookIterator, 3);
        
	/* Get the filename of the LRF file */
	int lastSlashIndex = bookListItem.Location.LastIndexOf("/");
	string onlyFilename = bookListItem.Location.Substring(lastSlashIndex + 1);
	string lrfFilename = librieFileStructure.LRFDirectory + Path.DirectorySeparatorChar + onlyFilename;
	
	if (! File.Exists(lrfFilename) ) {
	  MessageDialog mdLRFError = new MessageDialog (editLRFWindow, DialogFlags.DestroyWithParent, MessageType.Warning, 
						      ButtonsType.Close, 
						      "Could not locate the LRF file for the selected book. Unable to save XML information.");
	  mdLRFError.Run ();
	  mdLRFError.Destroy();
	  return;
	} else {
	  LRFFile lrfFile = new LRFFile(librieFileStructure.LRFDirectory + Path.DirectorySeparatorChar + onlyFilename);
	  lrfFile.XMLInfoBlock = xmlUnicodeBytes;
	  
	  /* Save the LRF file again */
	  FileStream fs = File.Create(lrfFilename);
	  byte[] lrfData = lrfFile.getAsBytes();
	  fs.Write(lrfData, 0, lrfData.Length);
	  fs.Close();      
	  
	  /* Create a new BookListItem from the modified LRF file */
	  BookListItem newBookListItem = new BookListItem(new LRFFile(lrfFilename), librieFileStructure.GetRelativeLocation(lrfFilename), false); 
	  ((ListStore)model).SetValue (this.selectedBookIterator, 3, newBookListItem);
	  
	  /* Refresh the Edit LRF Window with the new info */
	  thumbnailEditEntry.Text = DisplayString("");
	  authorEditEntry.Text = DisplayString(newBookListItem.Author);
	  authorReadingEditEntry.Text = DisplayString(newBookListItem.AuthorReading);
	  titleEditEntry.Text = DisplayString(newBookListItem.Title);
	  titleReadingEditEntry.Text = DisplayString(newBookListItem.TitleReading);
	  descriptionEditEntry.Text = DisplayString("");
	  creationDateEditEntry.Text = DisplayString(newBookListItem.Date);
	  publisherEditEntry.Text = DisplayString(newBookListItem.Publisher);
	  bookIDEditEntry.Text = DisplayString(newBookListItem.ID);
	  
	  XmlTextReader textReader = new XmlTextReader(new MemoryStream(lrfFile.XMLInfoBlock));
	  textReader.MoveToContent();
	  while (textReader.Read()) {
	    if ((textReader.NodeType == XmlNodeType.Element) && (! textReader.IsEmptyElement)) {
	      switch(textReader.Name) {
	      case "FreeText":
		descriptionEditEntry.Text = DisplayString(textReader.ReadElementString().Trim());
		break;
	      }
	    }
	  }
	  
	  /* Save the modified BookList and hide the window. Do not refresh the list, because we'd lose the iterator to the book */
	  updateBookList();
	  xmlEditWindow.Hide();
	}
      }
    }
	
    /* Called when browse button on LRF edit window is clicked */
    public void OnBrowseEditButtonClicked (object o, EventArgs args) {
      /* Open the file select window */
      FileSelection fs = new FileSelection ("Select Thumbnail Image");
      fs.SelectMultiple = false;
			
      /* Show the window, get the response. Set the text entry if a file is selected */
      fs.Show();
      ResponseType response = (ResponseType)fs.Run ();
      fs.Hide ();
      if (response == ResponseType.Cancel) {
	return;
      } else {
	thumbnailEditEntry.Text = DisplayString(fs.Filename);
      }
    }

    /* Called when the cancel button on the LRF edit window is clicked */
    public void OnCancelEditButtonClicked (object o, EventArgs args) {
      initializeList();
      editLRFWindow.Hide();
    }

    /* Called when the ok button on the LRF edit window is clicked */
    public void OnOKEditButtonClicked (object o, EventArgs args) {
      /* Get the selected book */
      TreeModel model = booklist.Model;
      BookListItem bookListItem = (BookListItem)((ListStore)model).GetValue (this.selectedBookIterator, 3);
        
      /* Set all the text fields for the book */
      bookListItem.Author = AcceptString(authorEditEntry.Text);
      bookListItem.AuthorReading = AcceptString(authorReadingEditEntry.Text);
      bookListItem.Title = AcceptString(titleEditEntry.Text);
      bookListItem.TitleReading = AcceptString(titleReadingEditEntry.Text);
      bookListItem.Date = creationDateEditEntry.Text;
      bookListItem.Publisher = AcceptString(publisherEditEntry.Text);
      bookListItem.ID = bookIDEditEntry.Text;
      
      byte[] thumbBytes = null;

      /* If a thumbnail file has been given, read it in and set the book value */
      string thumbnailFile = thumbnailEditEntry.Text;
      if ( thumbnailFile != "" ) {
	if (! File.Exists(thumbnailFile)) {
	  MessageDialog mdThumbnailError = new MessageDialog (editLRFWindow, DialogFlags.DestroyWithParent, MessageType.Error, 
							      ButtonsType.Close, "Thumbnail file does not exist");
	  mdThumbnailError.Run ();
	  mdThumbnailError.Destroy();
	  /* Blank the thumbnail, so that we don't try to set it in the LRF file */
	  thumbnailEditEntry.Text = DisplayString("");
	  thumbnailFile = "";
	} else {
	  /* Read the entire thumbnail file into thumbBytes */
	  thumbBytes = new byte[new FileInfo(thumbnailFile).Length];
	  byte[] tempBuffer = new byte[4096];
	  FileStream thumbfs = new FileStream(thumbnailFile, FileMode.Open);
	  int numBytesRead = 0;
	  int curPosition = 0;
	  while (0 != (numBytesRead = thumbfs.Read(tempBuffer, 0, 4096))) {
	    Buffer.BlockCopy(tempBuffer, 0, thumbBytes, curPosition, numBytesRead);
	    curPosition += numBytesRead;
	  }
	  thumbfs.Close ();
	  tempBuffer = null;
	  bookListItem.GIFData = thumbBytes;
	}
      }

      /* Get the filename of the LRF file */
      int lastSlashIndex = bookListItem.Location.LastIndexOf("/");
      string onlyFilename = bookListItem.Location.Substring(lastSlashIndex + 1);
      string lrfFilename = librieFileStructure.LRFDirectory + Path.DirectorySeparatorChar + onlyFilename;

      if (! File.Exists(lrfFilename) ) {
	MessageDialog mdLRFError = new MessageDialog (editLRFWindow, DialogFlags.DestroyWithParent, MessageType.Warning, 
						      ButtonsType.Close, 
						      "Could not locate the LRF file for the selected book. LRF file will not be updated.");
	mdLRFError.Run ();
	mdLRFError.Destroy();
      } else {
	LRFFile lrfFile = new LRFFile(librieFileStructure.LRFDirectory + Path.DirectorySeparatorChar + onlyFilename);

	/* Only set the thumbnail if one was specified */
	if ( thumbnailFile != "" ) {
	  lrfFile.GIFData = thumbBytes;
	}

	/* Go through the XML nodes, updating the values */
	MemoryStream outputStream = new MemoryStream();
	XmlTextWriter textWriter = new XmlTextWriter(outputStream, new System.Text.UnicodeEncoding());
	textWriter.Formatting = Formatting.Indented;
	textWriter.WriteStartDocument();
	textWriter.WriteStartElement("Info");
	textWriter.WriteAttributeString("version", "1.0");
	XmlTextReader textReader = new XmlTextReader(new MemoryStream(lrfFile.XMLInfoBlock));
	textReader.MoveToContent();
	    
	while (textReader.Read()) {
	  if ((textReader.NodeType == XmlNodeType.Element) && (! textReader.IsEmptyElement)) {
	    if (textReader.Name == "BookInfo") {
	      textWriter.WriteStartElement("BookInfo");
	      continue;
	    } else if (textReader.Name == "DocInfo") {
	      textWriter.WriteStartElement("DocInfo");
	      continue;
	    } else {
	      switch(textReader.Name) {
	      case "BookID":
		textWriter.WriteStartElement("BookID");
		textWriter.WriteString(bookListItem.ID);
		textWriter.WriteEndElement();
		break;
	      case "Author":
		textWriter.WriteStartElement("Author");
		textWriter.WriteAttributeString("reading", bookListItem.AuthorReading);
		textWriter.WriteString(bookListItem.Author);
		textWriter.WriteEndElement(); 
		break;
	      case "Title":
		textWriter.WriteStartElement("Title");
		textWriter.WriteAttributeString("reading", bookListItem.TitleReading);
		textWriter.WriteString(bookListItem.Title);
		textWriter.WriteEndElement(); 
		break;
	      case "Publisher":
		textWriter.WriteStartElement("Publisher");
		textWriter.WriteString(bookListItem.Publisher);
		textWriter.WriteEndElement(); 
		break;
	      case "CreationDate":
		textWriter.WriteStartElement("CreationDate");
		textWriter.WriteString(bookListItem.Date);
		textWriter.WriteEndElement(); 
		break;
	      case "FreeText":
		textWriter.WriteStartElement("FreeText");
		textWriter.WriteString(AcceptString(descriptionEditEntry.Text));
		textWriter.WriteEndElement(); 
		break;
	      default:
		textWriter.WriteNode(textReader, false);
		break;
	      }
	    }
	  } else if ((textReader.NodeType == XmlNodeType.EndElement) && (textReader.Name == "BookInfo") || (textReader.Name == "DocInfo" ))  {
	    textWriter.WriteNode(textReader, false);
	  }
	}
	
      textWriter.WriteEndDocument();
      textWriter.Close();
      textReader.Close();
      lrfFile.XMLInfoBlock = outputStream.GetBuffer();
  
      /* Save the LRF file again */
      FileStream fs = File.Create(lrfFilename);
      byte[] lrfData = lrfFile.getAsBytes();
      fs.Write(lrfData, 0, lrfData.Length);
      fs.Close();      
      }

      /* Refresh the display and save the BookList */
      updateBookList();
      initializeList();
      editLRFWindow.Hide();
    }

    /* Called when the Edit LRF window is deleted/closed */
    public void OnEditLRFWindowDeleteEvent (object o, DeleteEventArgs args) {
      initializeList();
      editLRFWindow.Hide();
      args.RetVal = true;
    }

    /* Called when a book is double clicked */
    public void OnBooklistRowActivated (object o, RowActivatedArgs args) {
      /* Get the selected book */
      TreeModel model = booklist.Model;
      TreeIter iter;
      ((ListStore)model).GetIter (out iter, args.Path);

      /* Remember the book that we've got selected. We'll need it to save the info later */
      this.selectedBookIterator = iter;

      BookListItem bookListItem = (BookListItem)((ListStore)model).GetValue (iter, 3);

      /* Populate the fields from the book */
      thumbnailEditEntry.Text = DisplayString("");
      authorEditEntry.Text = DisplayString(bookListItem.Author);
      authorReadingEditEntry.Text = DisplayString(bookListItem.AuthorReading);
      titleEditEntry.Text = DisplayString(bookListItem.Title);
      titleReadingEditEntry.Text = DisplayString(bookListItem.TitleReading);
      descriptionEditEntry.Text = DisplayString("");
      creationDateEditEntry.Text = DisplayString(bookListItem.Date);
      publisherEditEntry.Text = DisplayString(bookListItem.Publisher);
      bookIDEditEntry.Text = DisplayString(bookListItem.ID);

      int lastSlashIndex = bookListItem.Location.LastIndexOf("/");
      string onlyFilename = bookListItem.Location.Substring(lastSlashIndex + 1);

      if (! File.Exists(librieFileStructure.LRFDirectory + Path.DirectorySeparatorChar + onlyFilename)) {
	MessageDialog mdLRFError = new MessageDialog (editLRFWindow, DialogFlags.DestroyWithParent, MessageType.Warning, 
						      ButtonsType.Close, 
						      "Could not locate the LRF file for the selected book. LRF file will not be updated.");
	mdLRFError.Run ();
	mdLRFError.Destroy();
	descriptionEditEntry.Hide();
	descriptionEditLabel.Hide();
      } else {
	/* Open the LRF file, and read in the description */
	LRFFile lrfFile = new LRFFile(librieFileStructure.LRFDirectory + Path.DirectorySeparatorChar + onlyFilename);
	XmlTextReader textReader = new XmlTextReader(new MemoryStream(lrfFile.XMLInfoBlock));
	textReader.MoveToContent();
	while (textReader.Read()) {
	  if ((textReader.NodeType == XmlNodeType.Element) && (! textReader.IsEmptyElement)) {
	    switch(textReader.Name) {
	    case "FreeText":
	      descriptionEditEntry.Text = DisplayString(textReader.ReadElementString().Trim());
	      break;
	    }
	  }
	}
	descriptionEditEntry.Show();
	descriptionEditLabel.Show();
      }
      editLRFWindow.Show();
    }

    /* Called when BookList window is deleted/closed */
    public void OnBookListWindowDeleteEvent (object o, DeleteEventArgs args) {
      Application.Quit ();
      args.RetVal = true;
    }
		
    /* Called when the Add book button is pressed */
    public void OnAddButtonClicked(object o, EventArgs args) {
      /* Display the file selection dialog */
      FileSelection fs = new FileSelection ("Select LRF file(s) to add to the memory stick/BookList");
      fs.SelectMultiple = true;
      fs.ShowAll();
      fs.Complete("*.lrf");
      ResponseType response = (ResponseType)fs.Run ();
      fs.Hide ();
      if (response == ResponseType.Cancel) {
	return;
      }
          	
          	
      /* Add each file selected */
      foreach (string filename in fs.Selections) {
	string destFilename = librieFileStructure.LRFDirectory + Path.DirectorySeparatorChar + Path.GetFileName(filename);
	/* If the LRF file already exists, then prompt for overwrite and abort if user clicks No */
	if (File.Exists(destFilename)) {
	  MessageDialog md = new MessageDialog (bookListWindow, DialogFlags.DestroyWithParent, MessageType.Warning, ButtonsType.YesNo, 
						DisplayString(destFilename + " already exists. Do you want to overwrite?") );
	  ResponseType result = (ResponseType)md.Run ();
	  md.Destroy();
	  if (result == ResponseType.No) {
	    continue;
	  } else {
	    File.Delete(destFilename);
	  }
	}
				
	/* Copy the LRF file onto the memory stick and pop up an error if something goes wrong */
	try { 
	  File.Copy(filename, destFilename);
	} catch (Exception e) {
	  MessageDialog md = new MessageDialog (bookListWindow, DialogFlags.DestroyWithParent, MessageType.Error, ButtonsType.Close, 
						"Error Adding File: " + e.Message);
	  md.Run ();
	  md.Destroy();
	  return;
	}
          		
	/* Create the Book object, and get the thumbnail if it exists */
	BookListItem newBook;
	try {
	  newBook = new BookListItem(new LRFFile(destFilename), librieFileStructure.GetRelativeLocation(destFilename), false); 
	} catch (XMLInfoParseException e) {
	  // XML Info block in LRF file must be corrupt.
	  MessageDialog mdError = new MessageDialog (bookListWindow, DialogFlags.DestroyWithParent, MessageType.Error, ButtonsType.Close, 
						     DisplayString("Error occurred while parsing XML block in LRF file '" + filename + "'\n\nReason: " + e.Message + "\n\n The XML Editor will be opened so that the file can be corrected.") );
	  mdError.Run();
	  mdError.Destroy();

	  /* Open the LRF file, and read in the XML */
	  LRFFile lrfFile = new LRFFile(destFilename);

	  /* Fill the TextView in with the XML Information */
	  Gtk.TextBuffer xmlTextBuffer = new TextBuffer(null);
	  
	  /* Populate the text view box with the XML (minus the first two bytes if we're on Windows) */
	  if ((Environment.OSVersion.ToString().IndexOf("Unix")) != -1) {
	    xmlTextBuffer.Text = DisplayString(Encoding.Unicode.GetString(lrfFile.XMLInfoBlock));
	  } else {
	    xmlTextBuffer.Text = DisplayString(Encoding.Unicode.GetString(lrfFile.XMLInfoBlock).Substring(1));
	  }
	
	  xmlEditTextView.Buffer = xmlTextBuffer;
	  editingCorruptXML = true;
	  // Need to retain these, so that the edited XML can be saved to both.
	  corruptDestXMLFile = destFilename;
	  corruptXMLFile = filename;

	  xmlEditWindow.Show();

	  // Wait for the XML edit to finish
	  Application.Run();
	  
	  // Try to parse the XML again, because the XML edit is now finished.
	  try {
	    newBook = new BookListItem(new LRFFile(destFilename), librieFileStructure.GetRelativeLocation(destFilename), false); 
	  } catch (Exception f) {
	    MessageDialog mdFatalError = new MessageDialog (bookListWindow, DialogFlags.DestroyWithParent, MessageType.Error, ButtonsType.Close, 
							    DisplayString("Failed to add LRF file '" + filename + "'\n\nReason: " + f.Message) );
	    mdFatalError.Run();
	    mdFatalError.Destroy();
	    continue;
	  }
	} catch (Exception e) {
	  // Some unknown failure occurred.
	  MessageDialog mdError = new MessageDialog (bookListWindow, DialogFlags.DestroyWithParent, MessageType.Error, ButtonsType.Close, 
						     DisplayString("Error occurred while loading '" + filename + "': " + e.Message) );
	  mdError.Run();
	  mdError.Destroy();
	  continue;
	}
	Gdk.Pixbuf pb = null;
	if (newBook.GIFData.Length > 0) {
	  pb = new Gdk.Pixbuf(new MemoryStream(newBook.GIFData));
	}
            	
	/* Convert the LRF filesize to a readable MB/KB string */
	long fileSize = newBook.FileSize;
	string fileSizeDisplay;
	if (fileSize < (1024 * 1024)) {
	  fileSize = (fileSize / 1024);
	  fileSizeDisplay = fileSize.ToString() + " KB";
	} else {
	  fileSizeDisplay = String.Format("{0:F2} MB", ((float)fileSize / 1024.00 / 1024.00));
	}
            	
	/* String the a: off the LRF location */
	string location = newBook.Location;
	if (location.StartsWith("a:")) {
	  location = location.Substring(2);
	}
            	
	/* Add it in */
	((ListStore)(booklist.Model)).AppendValues(pb, DisplayString("<b>" + SecurityElement.Escape(newBook.Title) + "</b>\n" + SecurityElement.Escape(newBook.Author) + "\n" + SecurityElement.Escape(location)), 
						   fileSizeDisplay, newBook);
      }
          	
      /* Save the BookList to disk, and then update the display */
      updateBookList();
      initializeList();
    }        
       
                        
    /* Display preferences button has been pressed */
    public void OnPreferencesButtonClicked(object o, EventArgs args) {
      /* Set the widgets to their current configuration and show the preferences window */
      bookListSortOrder.SetHistory(sortType);
      librieRootEntry.Text = librieRootDir;
      preferencesWindow.Show();
    }
           
    /* Preferences cancel button has been pressed */
    public void OnPreferencesCancelButtonClicked(object o, EventArgs args) {
      /* Have we saved a configuration yet? */
      if (this.configSaved) {
	/* We've already saved, so we can just hide the preferences window */
	preferencesWindow.Hide();
      } else {
	/* We haven't yet saved, so we need to confirm that the user wants to close the preferences */
	MessageDialog md = new MessageDialog (bookListWindow, DialogFlags.DestroyWithParent, MessageType.Warning, ButtonsType.YesNo, 
					      "Preferences have not been saved. If you close this window, BookListGen will exit. Are you sure you want to quit without saving?" );
	ResponseType result = (ResponseType)md.Run ();
	md.Destroy();
	if (result == ResponseType.Yes) {
	  /* If the user wants to close without saving, then we'll have to terminate the GUI */
	  Application.Quit();
	}
      }
    }   
                                      
    /* Preferences window has been closed */
    public void OnPreferencesWindowDeleteEvent(object o, DeleteEventArgs args) {
      /* Have we saved a config yet? */
      if (this.configSaved) {
	/* We've saved, so we can just hide the preferences window */
	preferencesWindow.Hide();
	args.RetVal = true;
      } else {
	/* We haven't saved, but the user clicked the close icon, so we're going to close down */
	Application.Quit();
      }
    }        	
                                         
    /* Close button has been pressed */
    public void OnCloseButtonClicked(object o, EventArgs args) {
      /* Quit the application. There's nothing to do, because all changes are
	 automatically saved to disk as we go */
      Application.Quit();
    }
        
    /* Remove button has been pressed */
    public void OnRemoveButtonClicked(object o, EventArgs args) {
      TreeIter iter;
      TreeModel model;
        	
      /* Pop up a messagebox to confirm that the user wants to delete the book */
      MessageDialog md = new MessageDialog (bookListWindow, DialogFlags.DestroyWithParent, MessageType.Question,
					    ButtonsType.YesNo, "Really delete the " + booklist.Selection.CountSelectedRows() + 
					    " selected LRF file(s)? They will be deleted from the memory stick and removed from the BookList." );
      ResponseType result = (ResponseType)md.Run ();
      md.Destroy();
      if (result == ResponseType.No) {
	return;
      }
			
      /* Get all the selected rows, and store the iterators in a list */
      TreePath[] tps = booklist.Selection.GetSelectedRows(out model);
      ArrayList iters = new ArrayList();
      foreach (TreePath tp in tps) {
	((ListStore)model).GetIter (out iter, tp);
	iters.Add(iter);
      }
        	
      /* Go through the list of iterators and remove the items */
      foreach( TreeIter it in iters) {
	TreeIter i = it;
	BookListItem bookListItem = (BookListItem)((ListStore)model).GetValue (i, 3);
	int lastSlashIndex = bookListItem.Location.LastIndexOf("/");
	string onlyFilename = bookListItem.Location.Substring(lastSlashIndex + 1);

	if (! File.Exists(librieFileStructure.LRFDirectory + Path.DirectorySeparatorChar + onlyFilename)) {
	  /* If the LRF file has already been deleted, then show an error */
	  MessageDialog mdNotExistWarning = new MessageDialog (bookListWindow, DialogFlags.DestroyWithParent, MessageType.Warning, 
							       ButtonsType.Close, DisplayString("LRF File does not exist: " + librieFileStructure.LRFDirectory + Path.DirectorySeparatorChar + onlyFilename));
	  mdNotExistWarning.Run ();
	  mdNotExistWarning.Destroy();
	} else {
	  try {
	    /* Delete the LRF file, and pop up an error if something going wrong */
	    File.Delete(librieFileStructure.LRFDirectory + Path.DirectorySeparatorChar + onlyFilename);
	  } catch (Exception e) {
	    MessageDialog mdRemoveError = new MessageDialog (bookListWindow, DialogFlags.DestroyWithParent, MessageType.Error, 
							     ButtonsType.Close, "Error Deleting LRF File: " + e.Message);
	    mdRemoveError.Run ();
	    mdRemoveError.Destroy();
	    continue;
	  }
	}
				/* Remove the book from the list */
	((ListStore)model).Remove(ref i);
      }
    }
        
    /* A row in the BookList has been selected/deselected */
    private void OnSelectionChanged(object o, EventArgs args) {
      /* If something is selected, ungrey the remove button. If nothing is
	 selected, then grey it out */
      if (((TreeSelection)o).CountSelectedRows() > 0) {
	removeButton.Sensitive = true;
      } else {
	removeButton.Sensitive = false;
      }
    }
        
    /* A row in the BookList has been deleted */
    private void OnBookListRowDeleted(object o, Gtk.RowDeletedArgs args) {
      /* Only update if real time updates are enabled */
      if (realTimeUpdate) {
	updateBookList();
      }
    }
        
    /* A row in the BookList has been deleted */
    private void OnBookListOrderChanged(object o, Gtk.RowsReorderedArgs args) {
      /* Only update if real time updates are enabled */
      if (realTimeUpdate) {
	updateBookList();
      }
    }
         
    /* The refresh button has been clicked */
    private void OnRefreshButtonClicked(object o, EventArgs args) {
      MessageDialog mdFindLRFFiles = new MessageDialog (bookListWindow, DialogFlags.DestroyWithParent, MessageType.Question,
							ButtonsType.YesNo, "Do you want to add all LRF files on the memory stick to the BookList? " +
							"Selecting Yes will recreate the BookList from files on the memory stick. " +
							"Selecting No will reload the current BookList from the memory stick." );
      ResponseType result = (ResponseType)mdFindLRFFiles.Run ();
      mdFindLRFFiles.Destroy();

      // The user wants to find all LRF files and refresh the list using these files.
      if (result == ResponseType.Yes) {
	string booklistFilename = librieFileStructure.IndexFilename;
	OpenLibrie.BookList.BookList bookList = new OpenLibrie.BookList.BookList();
	
	ArrayList files = librieFileStructure.GetLRFFiles();
	foreach (string filename in files) {
	  bookList.AddItem(new BookListItem(new LRFFile(filename), librieFileStructure.GetRelativeLocation(filename), false));
	}

	/* Save the BookList, and pop up an error if it goes wrong */
	try {
	  bookList.SaveToFile(librieFileStructure.IndexFilename, false, false);
	} catch (Exception e) {
	  MessageDialog md = new MessageDialog (bookListWindow, DialogFlags.DestroyWithParent, MessageType.Error, ButtonsType.Close, 
					      "Error occurred while trying to write BookList: " + e.Message );
	  md.Run ();
	  md.Destroy();
	  return;
	}
      }

      initializeList();
    }
         
    /* Browse button on preferences window has been clicked */
    private void OnBrowseRootDirButtonClicked(object o, EventArgs args) {
      /* Open the directory select window */
      FileSelection fs = new FileSelection ("Select root directory of the memory stick");
      fs.SelectMultiple = false;
			
      /* We're only doing directories, so we hide the file operation buttons and file tree */
      fs.FileList.Parent.Hide();
      fs.SelectionEntry.Hide();
      fs.HideFileopButtons();
			
      /* Show the window, get the response. Set the text entry text if a directory is selected */
      fs.Show();
      ResponseType response = (ResponseType)fs.Run ();
      fs.Hide ();
      if (response == ResponseType.Cancel) {
	return;
      } else {
	librieRootEntry.Text = DisplayString(fs.Filename);
      }
    }
         
    /* Create filesystem button has been clicked */
    private void OnCreateFilesystemButtonClicked(object o, EventArgs args) {
      /* If the specified directory does not exist then show an error, put the focus back to the text entry widget and return */
      if (! Directory.Exists(librieRootEntry.Text)) {
	MessageDialog directorymd = new MessageDialog (bookListWindow, DialogFlags.DestroyWithParent, MessageType.Error, ButtonsType.Close, 
						       "The specified directory does not exist.");
	directorymd.Run ();
	directorymd.Destroy();
	librieRootEntry.GrabFocus();
	return;
      }
			
      /* Confirm that the user really does want to create a filesystem (Not that it is particularly dangerous. We're
	 just trying to be polite. */
      MessageDialog md = new MessageDialog (bookListWindow, DialogFlags.DestroyWithParent, MessageType.Warning,
					    ButtonsType.YesNo, "Are you sure that you want to create a filesystem on the memory stick?" );
      ResponseType result = (ResponseType)md.Run ();
      md.Destroy();
			
      /* If the user confirms, then create the filesystem. Pop up an error if something goes wrong */
      if (result == ResponseType.Yes) {
	try {
	  new LibrieFileStructure(librieRootEntry.Text).CreateDirStructure(false);
	  MessageDialog finishedmd = new MessageDialog (bookListWindow, DialogFlags.DestroyWithParent, MessageType.Info, ButtonsType.Ok, 
							"Filesystem created." );
	  finishedmd.Run ();
	  finishedmd.Destroy();
	} catch (Exception e) {
	  MessageDialog mderror = new MessageDialog (bookListWindow, DialogFlags.DestroyWithParent, MessageType.Error, ButtonsType.Close, 
						     "An error occurred while creating filesystem: " + e.Message );
	  mderror.Run ();
	  mderror.Destroy();
	}
      }
    }
        
    /* Ok button on preferences window has been clicked */
    private void OnPreferencesOkButtonClicked(object o, EventArgs args) {
      string rootLocation = librieRootEntry.Text;
			
      /* If the specified root directory doesn't exist then pop up an error, 
	 put focus back to the entry box and don't hide the preferences */
      if (! Directory.Exists(rootLocation)) {
	MessageDialog md = new MessageDialog (bookListWindow, DialogFlags.DestroyWithParent, MessageType.Error, ButtonsType.Close, "The specified directory does not exist.");
	md.Run ();
	md.Destroy();
	librieRootEntry.GrabFocus();
	return;
      }
			
      /* Set the current librie root to the specified one */
      LibrieFileStructure newLibrieFileStructure = new LibrieFileStructure(rootLocation);
			
      /* If there is no BookList in this directory, then prompt the user to create an empty one */
      if (! File.Exists(newLibrieFileStructure.IndexFilename)) {
	MessageDialog missingListMessBox = new MessageDialog (bookListWindow, DialogFlags.DestroyWithParent, MessageType.Warning, 
							      ButtonsType.YesNo, "BookList file is missing. Do you want to create an empty BookList?" );
	ResponseType missingListMessBoxResult = (ResponseType)missingListMessBox.Run ();
	missingListMessBox.Destroy();
	if (missingListMessBoxResult == ResponseType.No) {
	  /* If the user doesn't want to create it, then put focus back to the text entry box,
	     and don't close the preferences */
	  librieRootEntry.GrabFocus();
	  return;
	} else {
	  /* If the user wants an empty BookList file, just create an empty file. Pop up an error if something goes wrong */
	  try {
	    FileStream bookList = File.Create(newLibrieFileStructure.IndexFilename);
	    bookList.Close();
	  } catch (Exception e) {
	    MessageDialog bookListCreationErrorMessBox = new MessageDialog (bookListWindow, DialogFlags.DestroyWithParent, 
									    MessageType.Error, ButtonsType.Close, "Couldn't create the BookList (Maybe you need to create the filesystem first?)");
	    bookListCreationErrorMessBox.Run ();
	    bookListCreationErrorMessBox.Destroy();
	    librieRootEntry.GrabFocus();
	    return;
	  }
	}
      }
			
      /* If we've gotten this far, then we have a valid librie root directory, and we
	 have some BookList (even if it's empty). */
      if (this.configSaved == false) {
				/* If this is the first time we've saved a configuration, we need to
				   create the XML file and unhide the main BookList window */
	this.config.Create();
	this.configSaved = true;
	bookListWindow.Show();
      }
        	
      librieRootDir = rootLocation;
      librieFileStructure = newLibrieFileStructure;
			
      /* Set the sort type, and enable/disable the reorderable attribute of the BookList accordingly.
	 ( The list can only be reordered when the sort type is 'None' ) */
      switch (bookListSortOrder.History) {
      case 0:
        			/* Sort type is None */
	booklist.Reorderable = true;
	sortType = 0;
	break;
      case 1:
        			/* Sort type is Title */
	booklist.Reorderable = false;
	sortType = 1;
	break;
      case 2:
        			/* Sort type is Author */
	booklist.Reorderable = false;
	sortType = 2;
	break;
      default:
	break;
      };

      /* Save out configuration values */        	   	
      this.config.SetValue("//BookListGen//Preferences//set[@key='sortorder']", Convert.ToString(sortType));
      this.config.SetValue("//BookListGen//Preferences//set[@key='rootdir']", librieRootDir);

      /* Update the BookList and hide the preferences window */				
      initializeList();
      preferencesWindow.Hide();
    }   
        
  } /* Class: BookListGenGUI */
	
} /* Namespace: OpenLibrie */
