// LRFFile.cs
//
// Copyright (C) 2004-2005  Peter Knowles
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330,
// Boston, MA 02111-1307, USA.

using System;
using System.Text;
using System.IO;
using ICSharpCode.SharpZipLib.Zip.Compression;

namespace OpenLibrie.LRFLib {

  /// <summary>
  /// Represents a LRF file
  /// </summary>
  class LRFFile {

    /// <summary>
    /// Size of a WORD
    /// </summary>
    public const int WORD_SIZE = 2;

    /// <summary>
    /// Size of a DWORD
    /// </summary>
    public const int DWORD_SIZE = 4;

    /// <summary>
    /// Size of a QWORD
    /// </summary>
    public const int QWORD_SIZE = 8;

    /// <summary>
    /// Offset of length of uncompressed info block in LRF file
    /// </summary>
    public const int LRF_OFFSET_UNCOMXML_SIZE = 0x54;

    /// <summary>
    /// Offset of compressed info block data in LRF file
    /// </summary>
    public const int LRF_OFFSET_COMXML_DATA = LRF_OFFSET_UNCOMXML_SIZE + DWORD_SIZE;

    /// <summary>
    /// Offset of Header Signature
    /// </summary>
    public const int LRF_OFFSET_HEADER_SIGNATURE = 0x00;

    /// <summary>
    /// Offset of Header Version
    /// </summary>
    public const int LRF_OFFSET_HEADER_VERSION = 0x08;

    /// <summary>
    /// Offset of Header PseudoEncryption
    /// </summary>
    public const int LRF_OFFSET_HEADER_PSEUDOENCRYPTION = 0x0A;

    /// <summary>
    /// Offset of Header RootObjectID
    /// </summary>
    public const int LRF_OFFSET_HEADER_ROOTOBJECTID = 0x0C;

    /// <summary>
    /// Offset of Header NumObjects
    /// </summary>
    public const int LRF_OFFSET_HEADER_NUMOBJECTS = 0x10;

    /// <summary>
    /// Offset of Header ObjectIndexOffset
    /// </summary>
    public const int LRF_OFFSET_HEADER_OBJECTINDEXOFFSET = 0x18;

    /// <summary>
    /// Offset of Header Unknown1
    /// </summary>
    public const int LRF_OFFSET_HEADER_UNKNOWN1 = 0x20;

    /// <summary>
    /// Offset of Header Flags
    /// </summary>
    public const int LRF_OFFSET_HEADER_FLAGS = 0x24;

    /// <summary>
    /// Offset of Header Unknown2
    /// </summary>
    public const int LRF_OFFSET_HEADER_UNKNOWN2 = 0x25;

    /// <summary>
    /// Offset of Header Unknown3
    /// </summary>
    public const int LRF_OFFSET_HEADER_UNKNOWN3 = 0x26;

    /// <summary>
    /// Offset of Header Unknown4
    /// </summary>
    public const int LRF_OFFSET_HEADER_UNKNOWN4 = 0x28;

    /// <summary>
    /// Offset of Header Height
    /// </summary>
    public const int LRF_OFFSET_HEADER_HEIGHT = 0x2A;

    /// <summary>
    /// Offset of Header Width
    /// </summary>
    public const int LRF_OFFSET_HEADER_WIDTH = 0x2C;

    /// <summary>
    /// Offset of Header Unknown5
    /// </summary>
    public const int LRF_OFFSET_HEADER_UNKNOWN5 = 0x2E;

    /// <summary>
    /// Offset of Header Unknown6
    /// </summary>
    public const int LRF_OFFSET_HEADER_UNKNOWN6 = 0x2F;

    /// <summary>
    /// Offset of Header Unknown7
    /// </summary>
    public const int LRF_OFFSET_HEADER_UNKNOWN7 = 0x30;

    /// <summary>
    /// Offset of Header TOCObjectID
    /// </summary>
    public const int LRF_OFFSET_HEADER_TOCOBJECTID = 0x44;

    /// <summary>
    /// Offset of Header TOCObjectOffset
    /// </summary>
    public const int LRF_OFFSET_HEADER_TOCOBJECTOFFSET = 0x48;

    /// <summary>
    /// Offset of Header XMLCompSize
    /// </summary>
    public const int LRF_OFFSET_HEADER_COMXMLSIZE = 0x4C;

    /// <summary>
    /// Offset of Hearder Thumbnail Image Type
    /// </summary>
    public const int LRF_OFFSET_HEADER_THUMBIMAGETYPE = 0x4E;

    /// <summary>
    /// Offset of Header GIFSize
    /// </summary>
    public const int LRF_OFFSET_HEADER_GIFSIZE = 0x50;

    /// <summary>
    /// LRF Signature
    /// </summary>
    private byte[] lrfSignature;

    /// <summary>
    /// Keeps track of how much the header has grown/shrunk.
    /// Without this the offsets would get messed up.
    /// </summary>
    private int originalLRFObjectOffset;

    /// <summary>
    /// LRF Version
    /// </summary>
    private int version;

    /// <summary>
    /// LRF PsuedoEncryption
    /// </summary>
    private int pseudoEncryption;

    /// <summary>
    /// LRF RootObjectID
    /// </summary>
    private int rootObjectID;

    /// <summary>
    /// LRF NumObjects
    /// </summary>
    private Int64 numObjects;

    /// <summary>
    /// LRF ObjectIndexOffset
    /// </summary>
    private Int64 objectIndexOffset;

    /// <summary>
    /// LRF Unknown1
    /// </summary>
    private int unknown1;

    /// <summary>
    /// LRF Flags
    /// </summary>
    private byte flags;

    /// <summary>
    /// LRF Unknown2
    /// </summary>
    private byte unknown2;

    /// <summary>
    /// LRF Unknown3
    /// </summary>
    private int unknown3;

    /// <summary>
    /// LRF Unknown4
    /// </summary>
    private int unknown4;

    /// <summary>
    /// LRF Height
    /// </summary>
    private int height;

    /// <summary>
    /// LRF Width
    /// </summary>
    private int width;

    /// <summary>
    /// LRF Unknown5
    /// </summary>
    private byte unknown5;

    /// <summary>
    /// LRF Unknown6
    /// </summary>
    private byte unknown6;

    /// <summary>
    /// LRF Unknown7
    /// </summary>
    private byte[] unknown7;

    /// <summary>
    /// LRF TOCObjectID
    /// </summary>
    private int tocObjectID;

    /// <summary>
    /// LRF TOCObjectOffset
    /// </summary>
    private int tocObjectOffset;

    /// <summary>
    /// LRF XMLCompSize
    /// </summary>
    private int xmlCompSize;

    /// <summary>
    /// LRF Thumbnail Image Type
    /// </summary>
    private int thumbImageType;

    /// <summary>
    /// LRF GIFSize
    /// </summary>
    private int gifSize;

    /// <summary>
    /// Path of LRF file
    /// </summary>
    private string location;

    /// <summary>
    /// The LRF data objects (payload)
    /// </summary>
    private byte[] bytesLRFObjectData;

    /// <summary>
    /// The uncompressed UTF-16 info block
    /// </summary>
    private byte[] bytesXMLInfoBlock;

    /// <summary>
    /// The compressed UTF-16 info block
    /// </summary>
    private byte[] bytesCompXMLInfoBlock;

    /// <summary>
    /// The GIF file data for the thumnail image
    /// </summary>
    private byte[] bytesGIFData;

    /// <summary>
    /// The full path of the LRF file
    /// </summary>
    public string Location {
      get { return location; }
      set { location = value; }
    }

    /// <summary>
    /// The GIF data for the LRF thumbnail image
    /// </summary>
    public byte[] GIFData {
      get { return bytesGIFData; }
      set { 
	bytesGIFData = value;
	this.gifSize = bytesGIFData.Length;
	
	/* Only do this if we actually have a thumbnail image */
	if (this.gifSize > 0) {
	  /* Set the Image Type: 0x12 for PNG, 0x14 for GIF, 0x13 for BM and 0x10 for JPEG */
	  if ((bytesGIFData[1] == 'P') && (bytesGIFData[2] == 'N') && (bytesGIFData[3] == 'G')) {
	    /* It's a PNG */
	    this.thumbImageType = 0x12;
	  } else if ((bytesGIFData[0] == 'B') && (bytesGIFData[1] == 'M')) {
	    /* It's a BM */
	    this.thumbImageType = 0x13;
	  } else if ((bytesGIFData[6] == 'J') && (bytesGIFData[7] == 'F') && 
		     (bytesGIFData[8] == 'I') && (bytesGIFData[9] == 'F')) {
	    /* It's a JFIF */
	    this.thumbImageType = 0x11;
	  } else if ((bytesGIFData[0] == 'G') && (bytesGIFData[1] == 'I') && 
		     (bytesGIFData[2] == 'F') && (bytesGIFData[3] == '8')) {
	    /* It's a GIF */
	    this.thumbImageType = 0x14;
	  } else {
	    /* I give up, lets just call it a GIF and hope nobody notices */
	    this.thumbImageType = 0x14;
	  }
	}
      }
    }

    /// <summary>
    /// The file size of the LRF file
    /// </summary>
    private long fileSize;

    /// <summary>
    /// The file size of the LRF file
    /// </summary>
    public long FileSize {
      get { return fileSize; }
    }

    /// <summary>
    /// The uncompressed UTF-16 info block
    /// </summary>
    public byte[] XMLInfoBlock {
      get { return bytesXMLInfoBlock; }
      set { bytesXMLInfoBlock = value; updateCompXMLInfoBlock(); }
    }

    /// <summary>
    /// Updates the compressed XMLInfo Block
    /// </summary>
    private void updateCompXMLInfoBlock() {
      /* Allocate the required temporary memory for the Compressed Info Block */
      /* Yes, I know that this doesn't need to be 2, but I can't find the exact
         size right. */
      byte[] tempBytesCompXMLInfoBlock = new byte[bytesXMLInfoBlock.Length * 2];

      /* Compress the info block into the temporary array */
      Deflater d = new Deflater();
      d.SetInput(bytesXMLInfoBlock);
      d.Finish();
      /* Copy the size of the uncompressed block and the temporary array into the */
      /* permanent byte array */
      int length = d.Deflate(tempBytesCompXMLInfoBlock);
      this.bytesCompXMLInfoBlock = new byte[length + DWORD_SIZE];
      byte[] bytesUncompressedXMLSize = BitConverter.GetBytes(bytesXMLInfoBlock.Length);
      Buffer.BlockCopy(bytesUncompressedXMLSize, 0, bytesCompXMLInfoBlock, 0, DWORD_SIZE);
      Buffer.BlockCopy(tempBytesCompXMLInfoBlock, 0, bytesCompXMLInfoBlock, DWORD_SIZE, length);
      this.xmlCompSize = bytesCompXMLInfoBlock.Length;
    }

    /// <summary>
    /// Walks the object tree, and moves all offsets by a certain value
    /// </summary>
    /// <parameter name="lrfData">
    /// The object data
    /// </parameter>
    /// <parameter name="numObjects">
    /// The number of objects
    /// </parameter>
    /// <parameter name="objectIndexOffset">
    /// The offset to the object index.
    /// </parameter>
    /// <parameter name="diff">
    /// The adjustment value for each offset
    /// </parameter>
    private void relocateOffsets(byte[] lrfData, Int64 numObjects, Int64 objectIndexOffset, int diff) {
      for(int obNum=0; obNum<numObjects; obNum++) {
	Int32 oldOffset = BitConverter.ToInt32(lrfData, (int)(objectIndexOffset + (16 * obNum) + 4));
	byte[] bytesNewOffset = BitConverter.GetBytes(oldOffset + diff);
	Buffer.BlockCopy(bytesNewOffset, 0, lrfData, (int)(objectIndexOffset + (16 * obNum) + 4), DWORD_SIZE);
      }
    }

    /// <summary>
    /// Get the entire LRF file as an array of bytes.
    /// </summary>
    /// <returns>
    /// LRF file as array of bytes.
    /// </returns>
    public byte[] getAsBytes() {
      byte[] bytesHeader = generateHeader();
      byte[] lrfData = new byte[bytesHeader.Length + bytesCompXMLInfoBlock.Length + bytesGIFData.Length + bytesLRFObjectData.Length];

      /* Copy in the sections one after the other*/
      Buffer.BlockCopy(bytesHeader, 0, lrfData, 0, bytesHeader.Length);
      Buffer.BlockCopy(bytesCompXMLInfoBlock, 0, lrfData, bytesHeader.Length, bytesCompXMLInfoBlock.Length);
      Buffer.BlockCopy(bytesGIFData, 0, lrfData, bytesHeader.Length + bytesCompXMLInfoBlock.Length, bytesGIFData.Length);
      Buffer.BlockCopy(bytesLRFObjectData, 0, lrfData, 
      		       bytesHeader.Length + bytesCompXMLInfoBlock.Length + bytesGIFData.Length, bytesLRFObjectData.Length);

      int currentLRFObjectOffset = this.xmlCompSize + this.gifSize + bytesHeader.Length;
      int diff = currentLRFObjectOffset - originalLRFObjectOffset;
      relocateOffsets(lrfData, this.numObjects, BitConverter.ToInt64(bytesHeader, LRF_OFFSET_HEADER_OBJECTINDEXOFFSET), diff);

      return lrfData;
    }

    /// <summary>
    /// Generate the header
    /// </summary>
    /// <returns>
    /// bytes array representing the LRF header.
    /// </returns>
    private byte[] generateHeader() {
      byte[] header = new byte[84];

      /* Generate the signature */
      Buffer.BlockCopy(this.lrfSignature, 0, header, LRF_OFFSET_HEADER_SIGNATURE, 8);

      /* Generate the version */
      Byte[] bytesVersion = BitConverter.GetBytes(this.version);
      Buffer.BlockCopy(bytesVersion, 0, header, LRF_OFFSET_HEADER_VERSION, WORD_SIZE);

      /* Generate the PsuedoEncryption key */
      Byte[] bytesPseudoEncryption = BitConverter.GetBytes(this.pseudoEncryption);
      Buffer.BlockCopy(bytesPseudoEncryption, 0, header, LRF_OFFSET_HEADER_PSEUDOENCRYPTION, WORD_SIZE);

      /* Generate the RootObjectID */
      Byte[] bytesRootObjectID = BitConverter.GetBytes(this.rootObjectID);
      Buffer.BlockCopy(bytesRootObjectID, 0, header, LRF_OFFSET_HEADER_ROOTOBJECTID, DWORD_SIZE);

      /* Generate the NumberOfObjects */
      Byte[] bytesNumberOfObjects = BitConverter.GetBytes(this.numObjects);
      Buffer.BlockCopy(bytesNumberOfObjects, 0, header, LRF_OFFSET_HEADER_NUMOBJECTS, 8);

      /* Generate Unknown1 */
      Byte[] bytesUnknown1 = BitConverter.GetBytes(this.unknown1);
      Buffer.BlockCopy(bytesUnknown1, 0, header, LRF_OFFSET_HEADER_UNKNOWN1, DWORD_SIZE);

      /* Generate flags */
      header[LRF_OFFSET_HEADER_FLAGS] = this.flags;

      /* Generate Unknown2 */
      header[LRF_OFFSET_HEADER_UNKNOWN2] = this.unknown2;

      /* Generate Unknown3 */
      Byte[] bytesUnknown3 = BitConverter.GetBytes(this.unknown3);
      Buffer.BlockCopy(bytesUnknown3, 0, header, LRF_OFFSET_HEADER_UNKNOWN3, WORD_SIZE);

      /* Generate Unknown4 */
      Byte[] bytesUnknown4 = BitConverter.GetBytes(this.unknown4);
      Buffer.BlockCopy(bytesUnknown4, 0, header, LRF_OFFSET_HEADER_UNKNOWN4, WORD_SIZE);

      /* Generate Height */
      Byte[] bytesHeight = BitConverter.GetBytes(this.height);
      Buffer.BlockCopy(bytesHeight, 0, header, LRF_OFFSET_HEADER_HEIGHT, WORD_SIZE);

      /* Generate Width */
      Byte[] bytesWidth = BitConverter.GetBytes(this.width);
      Buffer.BlockCopy(bytesWidth, 0, header, LRF_OFFSET_HEADER_WIDTH, WORD_SIZE);

      /* Generate Unknown5 */
      header[LRF_OFFSET_HEADER_UNKNOWN5] = this.unknown5;

      /* Generate Unknown6 */
      header[LRF_OFFSET_HEADER_UNKNOWN6] = this.unknown6;

      /* Generate Unknown7 */
      Buffer.BlockCopy(this.unknown7, 0, header, LRF_OFFSET_HEADER_UNKNOWN7, 20);

      /* Generate the TOCObjectID */
      Byte[] bytesTocObjectID = BitConverter.GetBytes(this.tocObjectID);
      Buffer.BlockCopy(bytesTocObjectID, 0, header, LRF_OFFSET_HEADER_TOCOBJECTID, DWORD_SIZE);

      /* Generate the xmlCompSize */
      Byte[] bytesXMLCompSize = BitConverter.GetBytes(this.xmlCompSize);
      Buffer.BlockCopy(bytesXMLCompSize, 0, header, LRF_OFFSET_HEADER_COMXMLSIZE, WORD_SIZE);

      /* Generate Thumbnail Image Type */
      Byte[] bytesThumbImageType = BitConverter.GetBytes(this.thumbImageType);
      Buffer.BlockCopy(bytesThumbImageType, 0, header, LRF_OFFSET_HEADER_THUMBIMAGETYPE, WORD_SIZE);

      /* Generate the gifSize */
      Byte[] bytesGIFSize = BitConverter.GetBytes(this.gifSize);
      Buffer.BlockCopy(bytesGIFSize, 0, header, LRF_OFFSET_HEADER_GIFSIZE, DWORD_SIZE);

      int currentLRFObjectOffset = this.xmlCompSize + this.gifSize + header.Length;
      int diff = currentLRFObjectOffset - originalLRFObjectOffset;

      /* Generate the ObjectIndexOffset */
      this.objectIndexOffset = this.objectIndexOffset + diff;
      Byte[] bytesObjectIndexOffset = BitConverter.GetBytes(this.objectIndexOffset);
      Buffer.BlockCopy(bytesObjectIndexOffset, 0, header, LRF_OFFSET_HEADER_OBJECTINDEXOFFSET, 8);
      
      /* Generate the tocObjectOffset */
      this.tocObjectOffset = this.tocObjectOffset + diff;
      Byte[] bytesTocObjectOffset = BitConverter.GetBytes(this.tocObjectOffset);
      Buffer.BlockCopy(bytesTocObjectOffset, 0, header, LRF_OFFSET_HEADER_TOCOBJECTOFFSET, DWORD_SIZE);

      return header;
    }

    /// <summary>
    /// Parse the header from an LRF byte array
    /// </summary>
    private void parseHeader(byte[] lrfBytes) {
      /* Extract the signature */
      this.lrfSignature = new Byte[8];
      Buffer.BlockCopy(lrfBytes, LRF_OFFSET_HEADER_SIGNATURE, this.lrfSignature, 0, 8);

      /* Extract the version */
      this.version = BitConverter.ToInt16(lrfBytes, LRF_OFFSET_HEADER_VERSION);

      /* Extract the PsuedoEncryption key */
      this.pseudoEncryption = BitConverter.ToInt16(lrfBytes, LRF_OFFSET_HEADER_PSEUDOENCRYPTION);

      /* Extract the RootObjectID */
      this.rootObjectID = BitConverter.ToInt32(lrfBytes, LRF_OFFSET_HEADER_ROOTOBJECTID);

      /* Extract the NumberOfObjects */
      this.numObjects = BitConverter.ToInt64(lrfBytes, LRF_OFFSET_HEADER_NUMOBJECTS);

      /* Extract the ObjectIndexOffset */
      this.objectIndexOffset = BitConverter.ToInt64(lrfBytes, LRF_OFFSET_HEADER_OBJECTINDEXOFFSET);

      /* Extract Unknown1 */
      this.unknown1 = BitConverter.ToInt32(lrfBytes, LRF_OFFSET_HEADER_UNKNOWN1);

      /* Extract flags */
      this.flags = lrfBytes[LRF_OFFSET_HEADER_FLAGS];

      /* Extract Unknown2 */
      this.unknown2 = lrfBytes[LRF_OFFSET_HEADER_UNKNOWN2];

      /* Extract Unknown3 */
      this.unknown3 = BitConverter.ToInt16(lrfBytes, LRF_OFFSET_HEADER_UNKNOWN3);

      /* Extract Unknown4 */
      this.unknown4 = BitConverter.ToInt16(lrfBytes, LRF_OFFSET_HEADER_UNKNOWN4);

      /* Extract Height */
      this.height = BitConverter.ToInt16(lrfBytes, LRF_OFFSET_HEADER_HEIGHT);

      /* Extract Width */
      this.width = BitConverter.ToInt16(lrfBytes, LRF_OFFSET_HEADER_WIDTH);

      /* Extract Unknown5 */
      this.unknown5 = lrfBytes[LRF_OFFSET_HEADER_UNKNOWN5];

      /* Extract Unknown6 */
      this.unknown6 = lrfBytes[LRF_OFFSET_HEADER_UNKNOWN6];

      /* Extract Unknown7 */
      this.unknown7 = new Byte[20];
      Buffer.BlockCopy(lrfBytes, LRF_OFFSET_HEADER_UNKNOWN7, this.unknown7, 0, 20);

      /* Extract the TOCObjectID */
      this.tocObjectID = BitConverter.ToInt32(lrfBytes, LRF_OFFSET_HEADER_TOCOBJECTID);

      /* Extract the tocObjectOffset */
      this.tocObjectOffset = BitConverter.ToInt32(lrfBytes, LRF_OFFSET_HEADER_TOCOBJECTOFFSET);

      /* Extract the xmlCompSize */
      this.xmlCompSize = BitConverter.ToInt16(lrfBytes, LRF_OFFSET_HEADER_COMXMLSIZE);

      /* Extract Thumbnail Image Type */
      this.thumbImageType = BitConverter.ToInt16(lrfBytes, LRF_OFFSET_HEADER_THUMBIMAGETYPE);

      /* Extract the gifSize */
      this.gifSize = BitConverter.ToInt32(lrfBytes, LRF_OFFSET_HEADER_GIFSIZE);
    }

    /// <summary>
    /// Initializes a new instance of the LRFFile class, using data from the
    /// specified LRF file.
    /// </summary>
    /// <param name="filename">
    /// The name of the LRF file to read from.
    /// </param>
    public LRFFile(string filename) {
      this.location = filename;
      
      this.fileSize = new FileInfo(filename).Length;

      /* Used to store the whole LRF file as bytes */
      byte[] lrfBytes = new Byte[this.fileSize];

      #region ReadEntireLRFFile
      /* Read the entire LRF file into lrfBytes */
      byte[] tempBuffer = new Byte[4096];
      FileStream fs = new FileStream(filename, FileMode.Open);
      int numBytesRead = 0;
      int curPosition = 0;
      while (0 != (numBytesRead = fs.Read(tempBuffer, 0, 4096))) {
	Buffer.BlockCopy(tempBuffer, 0, lrfBytes, curPosition, numBytesRead);
	curPosition += numBytesRead;
      }
      fs.Close ();
      tempBuffer = null;
      #endregion

      /* Parse the entire header */
      parseHeader(lrfBytes);

      #region ReadCompressedInfoBlock
      /* Allocate the temporary memory required for the Compressed Info Block */
      int infoLength = BitConverter.ToInt16(lrfBytes, LRF_OFFSET_HEADER_COMXMLSIZE);
      byte[] bytesInfoData = new Byte[infoLength - DWORD_SIZE];

      /* Copy the Info block into the temporary buffer */
      Buffer.BlockCopy(lrfBytes, LRF_OFFSET_COMXML_DATA, bytesInfoData, 0, (infoLength - DWORD_SIZE));
      #endregion

      #region ReadGIFData
      /* Allocate the required memory for the GIF */      
      int gifLength = BitConverter.ToInt32(lrfBytes, LRF_OFFSET_HEADER_GIFSIZE);
      bytesGIFData = new Byte[gifLength];

      /* Copy the GIF data into bytesGIFData */
      Buffer.BlockCopy(lrfBytes, (LRF_OFFSET_UNCOMXML_SIZE + infoLength), bytesGIFData, 0, gifLength);
      #endregion

      #region ReadLRFObjectPayload
      /* Remember where we originally started */
      originalLRFObjectOffset = (LRF_OFFSET_UNCOMXML_SIZE + infoLength) + gifLength;

      /* Allocate the required memory for the data */
      int lrfDataOffset = (LRF_OFFSET_UNCOMXML_SIZE + infoLength) + gifLength;
      int lrfDataLength = (int)this.fileSize - lrfDataOffset;
      this.bytesLRFObjectData = new Byte[lrfDataLength];

      /* Copy the LRF data into bytesLRFObjectData */
      Buffer.BlockCopy(lrfBytes, lrfDataOffset, this.bytesLRFObjectData, 0, lrfDataLength);
      #endregion

      #region UncompressInfoBlock
      /* Allocate the required memory for the Uncompressed Info Block */
      int infoUncompressedLength = BitConverter.ToInt32(lrfBytes, LRF_OFFSET_UNCOMXML_SIZE);
      bytesXMLInfoBlock = new byte[infoUncompressedLength];

      /* Uncompress the info block */
      Inflater i = new Inflater();
      i.SetInput(bytesInfoData);
      i.Inflate(bytesXMLInfoBlock);
      #endregion

      updateCompXMLInfoBlock();

    }

  }
  
}
