// LibrieFileStructure.cs
//
// Copyright (C) 2004  Peter Knowles
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330,
// Boston, MA 02111-1307, USA.

using System;
using System.IO;
using System.Text.RegularExpressions;
using System.Collections;

namespace OpenLibrie.Utils {

  /// <summary>
  /// Represents the memory stick file structure for the Librie
  /// </summary>
  class LibrieFileStructure {

    /// <summary>
    /// Base path of file structure
    /// </summary>
    private string basePath;

	// <summary>
    /// Base path of file structure
    /// </summary>
    public string BasePath {
      get { return basePath; }
      set { basePath = value; }
    }
    
    // <summary>
    /// LRF directory
    /// </summary>
    public string LRFDirectory {
      get { 
      	char ps = Path.DirectorySeparatorChar;
      	return this.basePath + ps + "ebook" + ps + "bbeb" + ps + "book";
      }
    }
    
    /// <summary>
    /// Filename of index file
    /// </summary>
    public string IndexFilename {
      get {
      	char ps = Path.DirectorySeparatorChar;
      	return this.basePath + ps + "mssony" + ps + "librie" + ps + "booklist" + ps + "MgrTbl.dat";
      }
    }
    
    /// <summary>
    /// Initializes a default file structure class using specified base path
    /// </summary>
    /// <param name="path">
    /// Base path for file structure
    /// </param>
    public LibrieFileStructure(string path) {
      this.basePath = path.TrimEnd(Path.DirectorySeparatorChar);
    }
    
    /// <summary>
    /// Create the necessary directory structure for the Librie
    /// </summary>
    public void CreateDirStructure(Boolean verbose) {
      char ps = Path.DirectorySeparatorChar;
      /* Directory tree to create */
      string[] dirs = new string[] {
	(this.basePath), 
	(this.basePath + ps + "mssony"),
	(this.basePath + ps + "mssony" + ps + "librie"),
	(this.basePath + ps + "mssony" + ps + "librie" + ps + "booklist"),
	(this.basePath + ps + "mssony" + ps + "librie" + ps + "usb"),
	(this.basePath + ps + "mssony" + ps + "librie" + ps + "reader"),
	(this.basePath + ps + "mssony" + ps + "librie" + ps + "reader" + ps + "book"),
	(this.basePath + ps + "mssony" + ps + "librie" + ps + "reader" + ps + "book" + ps + "lrf"),
	(this.basePath + ps + "mssony" + ps + "librie" + ps + "reader" + ps + "book" + ps + "lrx"),
	(this.basePath + ps + "ebook"),
	(this.basePath + ps + "ebook" + ps + "bbeb"),
	(this.basePath + ps + "ebook" + ps + "bbeb" + ps + "book")
      };

      foreach (string dir in dirs) {
	if (! Directory.Exists(dir)) {
	  if (verbose) {
	    Console.WriteLine("Creating: {0}", dir);
	  }
	  Directory.CreateDirectory(dir);
	} else {
	  if (verbose) {
	    Console.WriteLine("Skipping: {0}", dir);
	  }
	}
      }
    }

    /// <summary>
    /// Gets a file's path relative to the root of the Librie's filesystem
    /// </summary>
    /// <param name="fullPath">
    /// The full path to calculate
    /// </param>
    /// <returns>
    /// The path relative to the root of the Librie's filesystem.
    /// </returns>
    public string GetRelativeLocation(string fullPath) {
      return fullPath.Substring(fullPath.IndexOf(this.basePath) + this.basePath.Length);
    }

    /// <summary>
    /// Gets list of LRF files in EBOOK/BBEB/BOOK directory
    /// </summary>
    /// <returns>
    /// ArrayList of LRF filenames
    /// </returns>
    public ArrayList GetLRFFiles() {
      /* Get all files in ebook directory */
      char ps = Path.DirectorySeparatorChar;
      string[] files = Directory.GetFiles(this.basePath + ps + "ebook" + ps + "bbeb" + ps + "book", "*.*");

      /* Filter the files for case-insensitive .lrf extension */
      ArrayList matchingFiles = new ArrayList();
      Regex lrfPattern = new Regex("\\.[lL][rR][fF]$");
      foreach (string filename in files) {
	if (lrfPattern.IsMatch(filename)) {
	  matchingFiles.Add(filename);
	}
      }
      files = null;
      /* Sort the LRF files into an alphabetical order */
      matchingFiles.Sort();
      return matchingFiles;
    }

  }
  
}
